import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";

// 1. Define your route arrays
const PROTECTED_ROUTES = ["/dashboard", "/profile", "/settings"];
const GUEST_ONLY_ROUTES = ["/login", "/register"];

export function middleware(req: NextRequest) {
  const { pathname } = req.nextUrl;
  
  // Check for the refresh_token (it's the most reliable indicator of a session)
  const token = req.cookies.get("refresh_token");

  // 2. Determine route category
  const isProtectedRoute = PROTECTED_ROUTES.some((route) => pathname.startsWith(route)) || pathname === "/";
  const isGuestRoute = GUEST_ONLY_ROUTES.includes(pathname);

  // 3. Logic: Redirect logged-in users away from Login/Register
  if (isGuestRoute && token) {
    return NextResponse.redirect(new URL("/dashboard", req.url));
  }

  // 4. Logic: Redirect logged-out users away from Protected Routes
  if (isProtectedRoute && !token) {
    return NextResponse.redirect(new URL("/login", req.url));
  }

  return NextResponse.next();
}

// 5. Config to ensure middleware doesn't run on static assets/images
export const config = {
  matcher: [
    /*
     * Match all paths except:
     * - _next/static (static files)
     * - _next/image (image optimization files)
     * - favicon.ico (favicon file)
     * - public folder files (svg, png, etc.)
     */
    '/((?!_next/static|_next/image|favicon.ico|.*\\.(?:svg|png|jpg|jpeg|gif|webp)$).*)',
  ],
};



// import { NextResponse } from "next/server";
// import type { NextRequest } from "next/server";

// export function middleware(req: NextRequest) {
//   const publicRoutes = ["/login", "/register", "/api/auth/login", "/api/auth/register"];
//   const token = req.cookies.get("refresh_token");
//   // console.log(publicRoutes.some(route => req.nextUrl.pathname.startsWith(route)))
//   // console.log(publicRoutes.includes(req.nextUrl.pathname))
//   // console.log(req.nextUrl.pathname)

//   if (!token && !req.nextUrl.pathname.startsWith("/register")) {
//     return NextResponse.redirect(new URL("/login", req.url));
//   }

//   return NextResponse.next();
// }
