"use client";
import { useState, useEffect } from "react";
import {
    Link2, Link2Off, Loader2, CheckCircle2,
    X, Globe, RefreshCcw, Search, Users, PlayCircle, Camera
} from "lucide-react";
import { cn } from "@/lib/utils";
import { useAuth } from "@/context/AuthContext";
import { Toast } from '@/components/ui/Toast';
import { authService } from "@/lib/auth";
// import { useAuth } from "@/context/AuthContext"
const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL;

interface AppConfig {
    id: string;
    name: string;
    icon: any; 
    description: string;
}

export default function ConnectedAppsSection() {
    const [connectedIds, setConnectedIds] = useState<string[]>([]);
    const [isInitialLoading, setIsInitialLoading] = useState(true);
    const [processingId, setProcessingId] = useState<string | null>(null);
    const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);
    const { user, refreshUser } = useAuth();
    const [availableApps, setAvailableApps] = useState<AppConfig[]>([]);

    const fetchData = async () => {
        setIsInitialLoading(true);
        try {
            // Fetch both available apps and current connections
            const res = await authService.getAvailableApps();
            if (res.success) {
                setAvailableApps(res.data);
            }
            
            if (user?.connected_apps) {
                setConnectedIds(user.connected_apps);
            }
        } catch (err) {
            showToast('error', "Failed to sync connection status.");
        } finally {
            setIsInitialLoading(false);
        }
    };

    useEffect(() => {
        fetchData();
    }, [user]);

    const handleReload = async ()=>{
        setIsInitialLoading(true);
        await refreshUser()
        fetchData()
        setIsInitialLoading(false);
    }
    const handleToggle = async (appId: string, isCurrentlyConnected: boolean) => {
        setProcessingId(appId);
        try {
            const result = isCurrentlyConnected
                ? await authService.disconnectApp(appId)
                : await authService.connectApp(appId);

            if (result.success) {
                setConnectedIds(prev =>
                    isCurrentlyConnected
                        ? prev.filter(id => id !== appId)
                        : [...prev, appId]
                );
                showToast('success', `Successfully ${isCurrentlyConnected ? 'disconnected' : 'connected'}!`);
            } else {
                showToast('error', `Request failed. ${result.message}`);
            }
        } catch (err) {
            console.error("Toggle App Error:", err);
            showToast('error', "An unexpected error occurred.");
        } finally {
            setProcessingId(null);
        }
    };

    const showToast = (type: 'success' | 'error', message: string) => {
        setToast({ type, message });
    };

    return (
        <div className="max-w-4xl mx-auto mt-12 px-6 pb-20">
            {/* Header Section */}
            <div className="flex justify-between items-start mb-10">
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">Connected Apps</h1>
                    <p className="text-muted-foreground mt-1">Manage and sync data with the Indzs ecosystem.</p>
                </div>
                <button
                    onClick={handleReload}
                    className="p-2 hover:bg-accent rounded-lg transition-colors border border-border"
                    disabled={isInitialLoading}
                >
                    <RefreshCcw size={18} className={cn(isInitialLoading && "animate-spin")} />
                </button>
            </div>

            <div className="grid gap-4">
                {isInitialLoading ? (
                    /* --- Skeleton Loading State --- */
                    Array(4).fill(0).map((_, i) => (
                        <div key={i} className="flex items-center justify-between p-5 rounded-2xl border border-border bg-muted/5 animate-pulse">
                            <div className="flex items-center gap-5">
                                <div className="w-12 h-12 rounded-xl bg-muted" />
                                <div className="space-y-2">
                                    <div className="h-4 w-32 bg-muted rounded" />
                                    <div className="h-3 w-48 bg-muted rounded" />
                                </div>
                            </div>
                            <div className="h-10 w-[130px] bg-muted rounded-lg" />
                        </div>
                    ))
                ) : availableApps.length === 0 ? (
                    /* --- Empty State --- */
                    <div className="flex flex-col items-center justify-center py-20 px-4 border-2 border-dashed border-border rounded-3xl bg-muted/5">
                        <div className="p-4 bg-muted rounded-full mb-4">
                            <Globe className="text-muted-foreground/50" size={32} />
                        </div>
                        <h3 className="text-lg font-semibold">No apps available</h3>
                        <p className="text-muted-foreground text-center max-w-xs mt-1">
                            We couldn't find any applications available for connection at this time.
                        </p>
                    </div>
                ) : (
                    /* --- Active List State --- */
                    availableApps.map((app) => {
                        const isConnected = connectedIds.includes(app.id);
                        return (
                            <div
                                key={app.id}
                                className={cn(
                                    "group flex items-center justify-between p-5 rounded-2xl border transition-all duration-300",
                                    isConnected ? "bg-card border-primary/20 shadow-sm" : "bg-muted/15 border-transparent opacity-80"
                                )}
                            >
                                <div className="flex items-center gap-5">
                                    <div className={cn(
                                        "w-12 h-12 rounded-xl flex items-center justify-center transition-all overflow-hidden",
                                        isConnected ? "bg-primary text-primary-foreground shadow-md" : "bg-background border border-border text-muted-foreground"
                                    )}>
                                        {typeof app.icon === 'string' ? (
                                             <img src={app.icon} alt="" className="w-full h-full object-cover"/>
                                        ) : (
                                            <app.icon size={24} />
                                        )}
                                    </div>
                                    <div>
                                        <h3 className="font-bold text-foreground flex items-center gap-2">
                                            {app.name}
                                            {isConnected && <CheckCircle2 size={14} className="text-primary" />}
                                        </h3>
                                        <p className="text-xs text-muted-foreground">{app.description}</p>
                                    </div>
                                </div>

                                <button
                                    disabled={processingId === app.id}
                                    onClick={() => handleToggle(app.id, isConnected)}
                                    className={cn(
                                        "min-w-[130px] h-10 rounded-lg font-bold text-xs uppercase tracking-wider transition-all flex items-center justify-center gap-2",
                                        isConnected
                                            ? "bg-secondary text-secondary-foreground hover:bg-destructive hover:text-white"
                                            : "bg-primary text-primary-foreground hover:shadow-lg hover:-translate-y-0.5"
                                    )}
                                >
                                    {processingId === app.id ? (
                                        <Loader2 size={16} className="animate-spin" />
                                    ) : isConnected ? (
                                        <><Link2Off size={16} /> Disconnect</>
                                    ) : (
                                        <><Link2 size={16} /> Link App</>
                                    )}
                                </button>
                            </div>
                        );
                    })
                )}
            </div>

            {/* Global Notification */}
            {toast && (
                <Toast
                    message={toast.message}
                    type={toast.type}
                    onClose={() => setToast(null)}
                />
            )}
        </div>
    );
}

// "use client";
// import { useState, useEffect } from "react";
// import {
//     Link2, Link2Off, Loader2, AlertCircle, CheckCircle2,
//     X, Globe, Tent, Gamepad2, Palette, RefreshCcw, Search, Users, PlayCircle, Camera
// } from "lucide-react";
// import { cn } from "@/lib/utils";
// import { useAuth } from "@/context/AuthContext"
// import { Toast } from '@/components/ui/Toast';
// import { authService } from "@/lib/auth"
// const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL
// // --- Types ---
// interface AppConfig {
//     id: string;
//     name: string;
//     icon: React.ElementType; // Using Lucide components
//     description: string;
// }

// const AVAILABLE_APPS: AppConfig[] = [
//     { 
//         id: "app_indzs", 
//         name: "Indzs Search", 
//         icon: Search, 
//         description: "Unified AI-powered search engine for the modern web." 
//     },
//     { 
//         id: "app_bhulok", 
//         name: "Bhulok", 
//         icon: Users, // Represents community/social network
//         description: "Connect with friends, share updates, and join global communities." 
//     },
//     { 
//         id: "app_chitraplay", 
//         name: "Chitraplay", 
//         icon: PlayCircle, // Represents video streaming/YouTube style
//         description: "Stream high-definition videos, movies, and original content." 
//     },
//     { 
//         id: "app_kujanam", 
//         name: "Kujanam", 
//         icon: Camera, // Or 'Camera' if Instagram icon isn't in your set
//         description: "Share your life through photos, reels, and visual storytelling." 
//     },
// ];

// export default function ConnectedAppsSection() {
//     const [connectedIds, setConnectedIds] = useState<string[]>([]);
//     const [isInitialLoading, setIsInitialLoading] = useState(true);
//     const [processingId, setProcessingId] = useState<string | null>(null);
//     const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);
//     const { user, loading } = useAuth();
//     const [availableApps, setAvailableApps ] = useState([])

//     const fetchAvailableApps = async() =>{
//         const res = await authService.getAvailableApps()
//         if(res.success){
//             setAvailableApps(res.data)
//         }
//     }
//     // 1. Fetch Connected Apps from Backend
//     const fetchConnections = async () => {
//         try {
//             const response = user?.connected_apps
//             setConnectedIds(response);
//         } catch (err) {
//             showToast('error', "Failed to sync connection status.");
//         } finally {
//             setIsInitialLoading(false);
//         }
//     };

//     useEffect(() => {
//         fetchAvailableApps()
//         fetchConnections();
//     }, []);

//     // 2. Connect/Disconnect Logic
//     // const handleToggle = async (appId: string, isCurrentlyConnected: boolean) => {
//     //     const endpoint = isCurrentlyConnected ? `/disconnect/${appId}/` : `/connect/${appId}/`;
//     //     setProcessingId(appId);

//     //     try {

//     //         const response = await fetch(`${API_URL}/api/accounts${endpoint}`, { method: 'POST', credentials: "include" });
//     //         const data = await response.json();
//     //         console.log(data)
//     //         if (!response.ok) {
//     //             showToast('error', `Request failed. ${data.error}`);
//     //             throw new Error(data.error);
//     //         }
//     //         setConnectedIds(prev =>
//     //             isCurrentlyConnected
//     //                 ? prev.filter(id => id !== appId)
//     //                 : [...prev, appId]
//     //         );

//     //         showToast('success', `Successfully ${isCurrentlyConnected ? 'disconnected' : 'connected'}!`);
//     //     } catch (err) {
//     //         // showToast('error', `Request failed. ${data.error}`);
//     //     } finally {
//     //         setProcessingId(null);
//     //     }
//     // };



//     // 2. Connect/Disconnect Logic
//     const handleToggle = async (appId: string, isCurrentlyConnected: boolean) => {
//         setProcessingId(appId);

//         try {
//             // Use the appropriate service method based on connection state
//             const result = isCurrentlyConnected
//                 ? await authService.disconnectApp(appId)
//                 : await authService.connectApp(appId);

//             if (result.success) {
//                 setConnectedIds(prev =>
//                     isCurrentlyConnected
//                         ? prev.filter(id => id !== appId)
//                         : [...prev, appId]
//                 );

//                 showToast('success', `Successfully ${isCurrentlyConnected ? 'disconnected' : 'connected'}!`);
//             } else {
//                 // result.message is automatically populated by handleResponse
//                 showToast('error', `Request failed. ${result.message}`);
//             }
//         } catch (err) {
//             console.error("Toggle App Error:", err);
//             showToast('error', "An unexpected error occurred.");
//         } finally {
//             setProcessingId(null);
//         }
//     };

//     const showToast = (type: 'success' | 'error', message: string) => {
//         setToast({ type, message });
//         // setTimeout(() => setToast(null), 4000);
//     };

//     return (
//         <div className="max-w-4xl mx-auto mt-12 px-6 pb-20">
//             <div className="flex justify-between items-start mb-10">
//                 <div>
//                     <h1 className="text-3xl font-bold tracking-tight">Connected Apps</h1>
//                     <p className="text-muted-foreground mt-1">Manage and sync data with the Indzs ecosystem.</p>
//                 </div>
//                 <button
//                     onClick={() => { setIsInitialLoading(true); fetchConnections(); }}
//                     className="p-2 hover:bg-accent rounded-lg transition-colors border border-border"
//                 >
//                     <RefreshCcw size={18} className={cn(isInitialLoading && "animate-spin")} />
//                 </button>
//             </div>

//             <div className="grid gap-4">
//                 {isInitialLoading ? (
//                     Array(4).fill(0).map((_, i) => (
//                         <div key={i} className="h-24 w-full bg-muted/20 animate-pulse rounded-2xl border border-border" />
//                     ))
//                 ) : (
//                         availableApps.map((app) => {
//                             const isConnected = connectedIds.includes(app.id);
//                             const Icon = app.icon;

//                             return (
//                                 <div
//                                     key={app.id}
//                                     className={cn(
//                                         "group flex items-center justify-between p-5 rounded-2xl border transition-all duration-300",
//                                         isConnected ? "bg-card border-primary/20 shadow-sm" : "bg-muted/5 border-transparent opacity-80"
//                                     )}
//                                 >
//                                     <div className="flex items-center gap-5">
//                                         <div className={cn(
//                                             "w-12 h-12 rounded-xl flex items-center justify-center transition-all",
//                                             isConnected ? "bg-primary text-primary-foreground shadow-md" : "bg-background border border-border text-muted-foreground"
//                                         )}>
//                                             {/* <Icon size={24} /> */}
//                                             <img src={app.icon} alt="app logo"className="w-full h-full"/>
//                                         </div>
//                                         <div>
//                                             <h3 className="font-bold text-foreground flex items-center gap-2">
//                                                 {app.name}
//                                                 {isConnected && <CheckCircle2 size={14} className="text-primary" />}
//                                             </h3>
//                                             <p className="text-xs text-muted-foreground">{app.description}</p>
//                                         </div>
//                                     </div>

//                                     <button
//                                         disabled={processingId === app.id}
//                                         onClick={() => handleToggle(app.id, isConnected)}
//                                         className={cn(
//                                             "min-w-[130px] h-10 rounded-lg font-bold text-xs uppercase tracking-wider transition-all flex items-center justify-center gap-2",
//                                             isConnected
//                                                 ? "bg-secondary text-secondary-foreground hover:bg-destructive hover:text-white"
//                                                 : "bg-primary text-primary-foreground hover:shadow-lg hover:-translate-y-0.5"
//                                         )}
//                                     >
//                                         {processingId === app.id ? (
//                                             <Loader2 size={16} className="animate-spin" />
//                                         ) : isConnected ? (
//                                             <><Link2Off size={16} /> Disconnect</>
//                                         ) : (
//                                                     <><Link2 size={16} /> Link App</>
//                                                 )}
//                                     </button>
//                                 </div>
//                             );
//                         })
//                     )}
//             </div>

//             {/* Global Notification */}
//             {/* {toast  && (
//                 <div className={cn(
//                     "fixed z-50 bottom-8 left-1/2 -translate-x-1/2 flex items-center gap-3 px-5 py-3.5 rounded-xl shadow-2xl border animate-in slide-in-from-bottom-5",
//                     toast.type === 'success' ? "bg-emerald-600 border-emerald-500 text-white" : "bg-destructive border-red-400 text-white"
//                 )}>
//                     {toast.type === 'success' ? <CheckCircle2 size={18} /> : <AlertCircle size={18} />}
//                     <span className="text-sm font-semibold tracking-wide">{toast.message}</span>
//                     <button onClick={() => setToast(null)} className="ml-2 hover:bg-white/20 rounded-full p-1 transition-colors">
//                         <X size={14} />
//                     </button>
//                 </div>
//             )} */}

//             {toast && (
//                 <Toast
//                     message={toast.message}
//                     type={toast.type}
//                     onClose={() => setToast(null)}
//                 />
//             )}
//         </div>
//     );
// }
