"use client";

import { useState } from "react";
import {
    ShieldCheck,
    Smartphone,
    Key,
    ArrowRight,
    CheckCircle2,
    Loader2,
    ChevronLeft,
    Lock,
    QrCode,
    Settings2,
    RefreshCcw
} from "lucide-react";
import { cn } from "@/lib/utils";
import { Toast } from '@/components/ui/Toast';

type MFAMethod = 'app' | 'sms' | 'backup_codes';

interface SetupState {
    method: MFAMethod | null;
    step: 'intro' | 'verify' | 'success' | 'manage';
}

export default function MultiFactorAuthPage() {
    // In a real app, 'isEnabled' and 'activeMethod' would come from your API
    const [isEnabled, setIsEnabled] = useState(true);
    const [activeMethod, setActiveMethod] = useState<MFAMethod>('app');

    const [loading, setLoading] = useState(false);
    const [setupState, setSetupState] = useState<SetupState>({
        method: null,
        step: isEnabled ? 'manage' : 'intro'
    });
    const [otp, setOtp] = useState("");
    const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);

    const handleToggleMFA = () => {
        setIsEnabled((prev)=>!prev);
        if (isEnabled) {
            // if (confirm("Disable 2-Step Verification? Your account will be less secure.")) {
                setSetupState({ method: null, step: 'intro' });
                setToast({ type: 'success', message: "MFA disabled" });
            // }
        } else {

            setSetupState({ method: null, step: 'intro' });
        }
    };

    const handleVerify = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);
        setTimeout(() => {
            setLoading(false);
            setIsEnabled(true);
            setActiveMethod(setupState.method || 'app');
            setSetupState(prev => ({ ...prev, step: 'success' }));
            setToast({ type: 'success', message: "MFA updated successfully!" });
        }, 1500);
    };

    return (
        <div className="max-w-3xl mx-auto pt-8 pb-20 px-4 animate-in fade-in duration-500">
            {/* Header */}
            <div className="mb-8 flex items-center gap-4">
                {setupState.step !== 'intro' && setupState.step !== 'manage' && (
                    <button onClick={() => setSetupState({ method: null, step: isEnabled ? 'manage' : 'intro' })}
                        className="p-2 hover:bg-accent rounded-full transition-colors">
                        <ChevronLeft size={20} />
                    </button>
                )}
                <div>
                    <h1 className="text-3xl font-bold tracking-tight">2-Step Verification</h1>
                    <p className="text-muted-foreground mt-1">Protect your account with an additional security layer.</p>
                </div>
            </div>

            {/* 1. STATUS CARD */}
            <div className={cn(
                "mb-8 p-6 rounded-2xl border transition-all flex items-center justify-between",
                isEnabled ? "bg-green-500/5 border-green-500/20" : "bg-card border-border shadow-sm"
            )}>
                <div className="flex items-center gap-4">
                    <div className={cn("p-3 rounded-xl", isEnabled ? "bg-green-500/10 text-green-600" : "bg-muted text-muted-foreground")}>
                        <ShieldCheck size={24} />
                    </div>
                    <div>
                        <h3 className="font-bold text-sm uppercase tracking-wider">Multi Factor Authentication  is {isEnabled ? "On" : "Off"}</h3>
                        <p className="text-xs text-muted-foreground">Since Jan 28, 2026</p>
                    </div>
                </div>
                <button
                    onClick={handleToggleMFA}
                    className={cn(
                        "px-4 py-2 rounded-lg text-xs font-bold uppercase tracking-wider transition-all",
                        isEnabled
                            ? "bg-secondary text-secondary-foreground hover:bg-destructive hover:text-white"
                            : "bg-primary text-primary-foreground hover:shadow-lg"
                    )}
                >
                    {isEnabled ? "Disable" : "Get Started"}
                </button>
            </div>

            {/* 2. MANAGEMENT VIEW (If Enabled) */}
            {isEnabled && setupState.step === 'manage' && (
                <div className="space-y-6">
                    <div className="bg-card border rounded-2xl overflow-hidden">
                        <div className="p-6 border-b bg-muted/30">
                            <h3 className="font-bold flex items-center gap-2">
                                <Settings2 size={18} /> Current Configuration
              </h3>
                        </div>

                        <div className="p-6 flex items-center justify-between">
                            <div className="flex items-center gap-4">
                                <div className="p-3 bg-primary/10 text-primary rounded-xl">
                                    {activeMethod === 'app' ? <QrCode size={24} /> : <Smartphone size={24} />}
                                </div>
                                <div>
                                    <h4 className="font-bold capitalize">{activeMethod.replace('_', ' ')}</h4>
                                    <p className="text-sm text-muted-foreground">Default verification method</p>
                                </div>
                            </div>
                            <button
                                onClick={() => setSetupState({ method: null, step: 'intro' })}
                                className="flex items-center gap-2 text-sm font-bold text-primary hover:bg-primary/5 px-4 py-2 rounded-lg transition-all"
                            >
                                <RefreshCcw size={16} /> Change Method
              </button>
                        </div>
                    </div>

                    <div className="p-4 bg-muted/20 border border-dashed rounded-xl flex items-center justify-between opacity-60">
                        <div className="flex items-center gap-3 text-sm">
                            <Key size={18} />
                            <span>Backup codes (8 remaining)</span>
                        </div>
                        <button className="text-xs font-bold hover:underline">View</button>
                    </div>
                </div>
            )}

            {/* 3. SELECTION LIST (Intro) */}
            {!isEnabled || setupState.step === 'intro' ? (
                <div className="space-y-4 animate-in slide-in-from-bottom-2">
                    <h2 className="text-sm font-bold uppercase tracking-widest text-muted-foreground ml-1">
                        {isEnabled ? "Switch to a different method" : "Choose a setup method"}
                    </h2>
                    <MethodCard
                        icon={<QrCode size={22} />}
                        title="Authenticator App"
                        isActive={activeMethod === 'app' && isEnabled}
                        description="Get codes from an app like Google Authenticator."
                        onSelect={() => setSetupState({ method: 'app', step: 'verify' })}
                    />
                    <MethodCard
                        icon={<Smartphone size={22} />}
                        title="SMS Verification"
                        isActive={activeMethod === 'sms' && isEnabled}
                        description="Codes sent via text message to your phone."
                        onSelect={() => setSetupState({ method: 'sms', step: 'verify' })}
                    />
                </div>
            ) : null}

            {/* 4. VERIFICATION STEP */}
            {setupState.step === 'verify' && (
                <div className="bg-card border rounded-2xl p-8 text-center animate-in zoom-in-95">
                    <div className="w-20 h-20 bg-primary/10 text-primary rounded-full flex items-center justify-center mx-auto mb-6">
                        <Lock size={32} />
                    </div>
                    <h3 className="text-xl font-bold">Verify it's you</h3>
                    <p className="text-sm text-muted-foreground mt-2 mb-8">
                        Enter the code sent to your {setupState.method === 'app' ? 'Authenticator App' : 'Phone'}.
           </p>
                    <form onSubmit={handleVerify} className="max-w-xs mx-auto space-y-4">
                        <input
                            autoFocus
                            type="text"
                            maxLength={6}
                            value={otp}
                            onChange={(e) => setOtp(e.target.value)}
                            className="w-full h-14 text-center text-2xl font-mono border rounded-xl focus:ring-2 focus:ring-primary outline-none"
                            placeholder="000000"
                        />
                        <button disabled={otp.length < 6 || loading} className="w-full h-12 bg-primary text-primary-foreground rounded-xl font-bold flex items-center justify-center gap-2">
                            {loading ? <Loader2 className="animate-spin" /> : "Confirm Change"}
                        </button>
                    </form>
                </div>
            )}

            {/* 5. SUCCESS STEP */}
            {setupState.step === 'success' && (
                <div className="bg-card border rounded-2xl p-12 text-center">
                    <div className="inline-flex p-4 bg-green-500/10 text-green-600 rounded-full mb-6">
                        <CheckCircle2 size={48} />
                    </div>
                    <h3 className="text-2xl font-bold">Security Updated</h3>
                    <p className="text-muted-foreground mt-2">Your 2-Step Verification method has been changed.</p>
                    <button onClick={() => setSetupState({ method: null, step: 'manage' })} className="mt-8 px-8 py-3 bg-primary text-primary-foreground rounded-xl font-bold transition-all">
                        Finish
            </button>
                </div>
            )}

            {toast && <Toast message={toast.message} type={toast.type} onClose={() => setToast(null)} />}
        </div>
    );
}

function MethodCard({ icon, title, description, onSelect, isActive }) {
    return (
        <button onClick={onSelect} className={cn(
            "w-full p-6 flex items-center justify-between bg-card border rounded-2xl hover:bg-accent/30 transition-all text-left group",
            isActive && "border-primary bg-primary/5"
        )}>
            <div className="flex gap-5">
                <div className={cn("p-3 bg-muted rounded-xl text-muted-foreground group-hover:text-primary transition-colors", isActive && "text-primary bg-primary/10")}>
                    {icon}
                </div>
                <div>
                    <div className="flex items-center gap-2">
                        <h4 className="font-bold text-base">{title}</h4>
                        {isActive && <span className="text-[10px] bg-primary text-primary-foreground px-2 py-0.5 rounded-full uppercase">Current</span>}
                    </div>
                    <p className="text-sm text-muted-foreground">{description}</p>
                </div>
            </div>
            <ArrowRight size={18} className="text-muted-foreground group-hover:translate-x-1 transition-transform" />
        </button>
    );
}

// "use client";

// import { useState } from "react";
// import {
//   ShieldCheck,
//   Smartphone,
//   Key,
//   Mail,
//   ArrowRight,
//   CheckCircle2,
//   Trash2,
//   Plus,
//   Loader2,
//   ChevronLeft,
//   AlertTriangle,
//   Lock,
//   QrCode
// } from "lucide-react";
// import { cn } from "@/lib/utils";
// import { Toast } from '@/components/ui/Toast';

// // Mock types for UI states
// type MFAMethod = 'app' | 'sms' | 'backup_codes';

// interface SetupState {
//   method: MFAMethod | null;
//   step: 'intro' | 'verify' | 'success';
// }

// export default function MultiFactorAuthPage() {
//   const [isEnabled, setIsEnabled] = useState(true);
//   const [loading, setLoading] = useState(false);
//   const [setupState, setSetupState] = useState<SetupState>({ method: null, step: 'intro' });
//   const [otp, setOtp] = useState("");
//   const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);

//   // Toggle MFA entirely
//   const handleToggleMFA = () => {
//     if (isEnabled) {
//       if (confirm("Are you sure you want to disable MFA? This makes your account less secure.")) {
//         setIsEnabled(false);
//         setToast({ type: 'success', message: "2-Step Verification turned off" });
//       }
//     } else {
//       setSetupState({ method: 'app', step: 'intro' });
//     }
//   };

//   const startSetup = (method: MFAMethod) => {
//     setSetupState({ method, step: 'verify' });
//   };

//   const handleVerify = async (e: React.FormEvent) => {
//     e.preventDefault();
//     setLoading(true);

//     // Simulate API call
//     setTimeout(() => {
//       setLoading(false);
//       setSetupState(prev => ({ ...prev, step: 'success' }));
//       setIsEnabled(true);
//       setToast({ type: 'success', message: "MFA setup successfully!" });
//     }, 1500);
//   };

//   const resetSetup = () => {
//     setSetupState({ method: null, step: 'intro' });
//     setOtp("");
//   };

//   return (
//     <div className="max-w-3xl mx-auto pt-8 pb-20 px-4 animate-in fade-in duration-500">
//       {/* Header */}
//       <div className="mb-8 flex items-center gap-4">
//         {setupState.step !== 'intro' && (
//             <button onClick={resetSetup} className="p-2 hover:bg-accent rounded-full transition-colors">
//                 <ChevronLeft size={20} />
//             </button>
//         )}
//         <div>
//           <h1 className="text-3xl font-bold tracking-tight">2-Step Verification</h1>
//           <p className="text-muted-foreground mt-1">
//             Add an extra layer of security to your account by requiring more than just a password.
//           </p>
//         </div>
//       </div>

//       {/* Status Card */}
//       <div className={cn(
//         "mb-8 p-6 rounded-2xl border transition-all flex items-center justify-between",
//         isEnabled ? "bg-green-500/5 border-green-500/20" : "bg-card border-border shadow-sm"
//       )}>
//         <div className="flex items-center gap-4">
//           <div className={cn(
//             "p-3 rounded-xl",
//             isEnabled ? "bg-green-500/10 text-green-600" : "bg-muted text-muted-foreground"
//           )}>
//             <ShieldCheck size={24} />
//           </div>
//           <div>
//             <h3 className="font-bold">Status: {isEnabled ? "Active" : "Off"}</h3>
//             <p className="text-xs text-muted-foreground">
//                 {isEnabled ? "Your account is protected with MFA" : "MFA is currently disabled"}
//             </p>
//           </div>
//         </div>
        // <button
        //   onClick={handleToggleMFA}
        //   className={cn(
        //     "px-4 py-2 rounded-lg text-xs font-bold uppercase tracking-wider transition-all",
        //     isEnabled 
        //       ? "bg-secondary text-secondary-foreground hover:bg-destructive hover:text-white" 
        //       : "bg-primary text-primary-foreground hover:shadow-lg"
        //   )}
        // >
        //   {isEnabled ? "Disable" : "Get Started"}
        // </button>
//       </div>

//       {/* Dynamic Content Area */}
//       {setupState.step === 'intro' ? (
//         <div className="space-y-4">
//           <h2 className="text-sm font-bold uppercase tracking-widest text-muted-foreground ml-1">Available Methods</h2>

//           <MethodCard 
//             icon={<QrCode size={22} />}
//             title="Authenticator App"
//             description="Use apps like Google Authenticator or Authy to generate codes."
//             onSelect={() => startSetup('app')}
//           />

//           <MethodCard 
//             icon={<Smartphone size={22} />}
//             title="SMS Verification"
//             description="Receive a 6-digit code via text message to your phone."
//             onSelect={() => startSetup('sms')}
//           />

//           <MethodCard 
//             icon={<Key size={22} />}
//             title="Backup Codes"
//             description="One-time use codes for when you don't have your phone."
//             onSelect={() => startSetup('backup_codes')}
//           />
//         </div>
//       ) : setupState.step === 'verify' ? (
//         <div className="bg-card border rounded-2xl overflow-hidden shadow-sm animate-in slide-in-from-bottom-4">
//           <div className="p-8 flex flex-col items-center text-center">
//             <div className="w-48 h-48 bg-white p-4 rounded-xl border-4 border-muted mb-6">
//                {/* Mock QR Code */}
//                <div className="w-full h-full bg-slate-100 flex items-center justify-center text-slate-300">
//                   <QrCode size={80} />
//                </div>
//             </div>

//             <h3 className="text-xl font-bold">Scan the QR Code</h3>
//             <p className="text-sm text-muted-foreground max-w-sm mt-2">
//               Open your authenticator app and scan the code above. Then, enter the 6-digit code provided by the app.
//             </p>

//             <form onSubmit={handleVerify} className="w-full max-w-xs mt-8 space-y-4">
//               <input 
//                 type="text" 
//                 maxLength={6}
//                 placeholder="000 000"
//                 value={otp}
//                 onChange={(e) => setOtp(e.target.value)}
//                 className="w-full h-14 text-center text-2xl font-mono tracking-[0.5em] border rounded-xl focus:ring-2 focus:ring-primary outline-none transition-all"
//               />
//               <button 
//                 disabled={otp.length < 6 || loading}
//                 className="w-full h-12 bg-primary text-primary-foreground rounded-xl font-bold flex items-center justify-center gap-2 disabled:opacity-50"
//               >
//                 {loading ? <Loader2 className="animate-spin" /> : "Verify and Activate"}
//               </button>
//             </form>
//           </div>
//         </div>
//       ) : (
//         <div className="bg-card border rounded-2xl p-12 text-center animate-in zoom-in-95">
//             <div className="inline-flex p-4 bg-green-500/10 text-green-600 rounded-full mb-6">
//                 <CheckCircle2 size={48} />
//             </div>
//             <h3 className="text-2xl font-bold">You're all set!</h3>
//             <p className="text-muted-foreground mt-2 max-w-sm mx-auto">
//                 2-Step verification is now active. Your account is significantly more secure.
//             </p>
//             <button 
//                 onClick={resetSetup}
//                 className="mt-8 px-8 py-3 bg-muted hover:bg-accent rounded-xl font-bold transition-all"
//             >
//                 Back to Settings
//             </button>
//         </div>
//       )}

//       {/* Safety Alert */}
//       <div className="mt-12 p-6 border rounded-2xl bg-blue-500/5 border-blue-500/20 flex gap-4">
//         <Lock className="text-blue-600 shrink-0" size={24} />
//         <div>
//           <h4 className="text-sm font-bold text-blue-700">Why use 2-Step Verification?</h4>
//           <p className="text-xs text-blue-600/80 leading-relaxed mt-1">
//             Even if someone steals your password, they won't be able to log in without access 
//             to your physical device or backup codes.
//           </p>
//         </div>
//       </div>

//       {toast && (
//         <Toast
//           message={toast.message}
//           type={toast.type}
//           onClose={() => setToast(null)}
//         />
//       )}
//     </div>
//   );
// }

// // Sub-component for Method Selection
// function MethodCard({ icon, title, description, onSelect }) {
//     return (
//         <button 
//             onClick={onSelect}
//             className="w-full p-6 flex items-center justify-between bg-card border rounded-2xl hover:bg-accent/30 transition-all text-left group"
//         >
//             <div className="flex gap-5">
//                 <div className="p-3 bg-muted rounded-xl text-muted-foreground group-hover:bg-primary/10 group-hover:text-primary transition-colors">
//                     {icon}
//                 </div>
//                 <div>
//                     <h4 className="font-bold text-base">{title}</h4>
//                     <p className="text-sm text-muted-foreground">{description}</p>
//                 </div>
//             </div>
//             <ArrowRight size={18} className="text-muted-foreground group-hover:translate-x-1 transition-transform" />
//         </button>
//     )
// }