"use client";
import { useState, useEffect } from "react";
import Link from "next/link";
import { ArrowLeft, Eye, EyeOff, AlertCircle, ShieldQuestion, Loader2, CheckCircle2 } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateForgotPass, validateIdentifier } from "@/lib/auth-validation";
import { useRouter } from "next/navigation";
import { Logo } from "@/components/ui/Logo"
const BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL
export default function ForgotPassword() {
  const [step, setStep] = useState(1); // 1: Identifier, 2: OTP, 3: Manual Recovery Challenge, 4: Reset
  const [showPass, setShowPass] = useState(false);
  const [error, setError] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const router = useRouter()
  const [countdown, setCountdown] = useState(5);
  const [formData, setFormData] = useState({
    identifier: "",
    otp: "",
    password: "",
    confirmPassword: ""
  });

  const [resendCountdown, setResendCountdown] = useState(0);
  const [canResend, setCanResend] = useState(true);

  // Handle Resend OTP Timer
  useEffect(() => {
    let timer: NodeJS.Timeout;
    if (resendCountdown > 0) {
      setCanResend(false);
      timer = setInterval(() => setResendCountdown((prev) => prev - 1), 1000);
    } else {
      setCanResend(true);
    }
    return () => clearInterval(timer);
  }, [resendCountdown]);

  // Handle Redirection Timer
  useEffect(() => {
    let timer: NodeJS.Timeout;
    if (step === 5 && countdown > 0) {
      timer = setTimeout(() => setCountdown(countdown - 1), 1000);
    } else if (step === 5 && countdown === 0) {
      router.push("/login");
    }
    return () => clearTimeout(timer);
  }, [step, countdown, router]);


  // Step 1: Submit Identifier
  const handleIdentify = async (e: React.FormEvent) => {
    e.preventDefault();
    const vError = validateIdentifier(formData.identifier);
    if (vError) return setError(vError);

    setIsLoading(true);
    setError("");
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/forgot/send-otp/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier }),
      });
      const data = await res.json()
      if (!res.ok) { throw new Error(data.error) };

      setStep(2);
    } catch (err: any) {
      setError(err.message || "Something Went Wrong!");
    } finally {
      setIsLoading(false);
    }
  };

  // Step 3: Manual Recovery (Blind Verification)
  const handleManualChallenge = async (e: React.FormEvent) => {
    e.preventDefault();
    const vError = validateIdentifier(formData.identifier);
    if (vError) return setError(vError);

    setIsLoading(true);
    setError("");
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/forgot/send-otp/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          identifier: formData.identifier,
        }),
      });

      const data = await res.json()
      if (!res.ok) { throw new Error(data.error) };
      setStep(2);
      setFormData(prev => ({ ...prev, otp: "" }));
    } catch (err: any) {
      setError(err.message || "Something Went Wrong!");
    } finally {
      setIsLoading(false);
    }
  };

  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    console.log(formData.identifier)
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/forgot/verify-otp/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier, otp: formData.otp }),
      });
      const data = await res.json()
      if (!res.ok) { throw new Error(data.error) };
      setStep(4);
    } catch (err: any) {
      setError(err.message || "Something Went Wrong");
    } finally {
      setIsLoading(false);
    }
  };

  const handleReset = async (e: React.FormEvent) => {
    e.preventDefault();
    const vError = validateForgotPass(3, formData);
    if (vError) return setError(vError);

    setIsLoading(true);
    try {
      const res = await fetch(`${BASE_URL}/api/auth/password/reset/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ identifier: formData.identifier, new_password: formData.password }),
      });
      const data = await res.json()
      if (!res.ok) { throw new Error(data.error) };
      setStep(5);
      // console.log("Success");
      // Redirect or show success state here
    } catch (err: any) {
      setError(err.message || "Something Went Wrong!");
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(()=>{
    setError(null)
  },[step])

  return (
    <div className="min-h-screen bg-background flex flex-col items-center justify-center md:p-4 w-full">
      {/* <div className="w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] border border-border shadow-2xl overflow-hidden flex flex-col md:flex-row min-h-[500px]"> */}
      <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] overflow-hidden flex flex-col md:flex-row min-h-[500px] transition-all duration-500">
        {/* Left Side: Contextual Info (Identical Design) */}
        <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
          <div className="space-y-6">
            <Logo />
            <div className="space-y-2">
              <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                {step === 1 && "Account recovery"}
                {step === 2 && "Check your phone"}
                {step === 3 && "Verify Identity"}
                {step === 4 && "Change password"}
                {step === 5 && "Success!"}

              </h1>
              <p className="text-muted-foreground text-lg font-light max-w-sm">
                {step === 1 && "To help keep your account secure, Indzs needs to confirm this account belongs to you."}
                {step === 2 && `Enter the verification code sent to ${formData.identifier}`}
                {step === 3 && "Enter the recovery email or phone number associated with this account."}
                {step === 4 && "Create a new, strong password that you don't use for other websites."}
                {step === 5 && `Your password has been reset. Redirecting to login in ${countdown}s...`}
              </p>
            </div>
          </div>
          {step > 1 && (
            <button onClick={() => setStep(step - 1)} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all">
              <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
            </button>
          )}
        </div>

        {/* Right Side: Inputs (Identical Design) */}
        <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
          <form onSubmit={step === 1 ? handleIdentify : step === 2 ? handleVerifyOTP : step === 3 ? handleManualChallenge : handleReset} className="space-y-6 animate-in fade-in slide-in-from-right-4 duration-500">

            {/* Step 1: Identity */}
            {step === 1 && (
              <div className="relative group">
                <input
                  type="text" required
                  className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 text-foreground outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                  placeholder="Email or phone"
                  value={formData.identifier}
                  onChange={(e) => setFormData({ ...formData, identifier: e.target.value })}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all pointer-events-none peer-focus:-top-3 peer-focus:left-3 peer-focus:text-sm peer-focus:text-sidebar-primary peer-focus:bg-card peer-focus:px-2 peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:left-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card peer-[:not(:placeholder-shown)]:px-2">
                  Email or phone
                </label>
              </div>
            )}

            {/* Step 2: Verification Code */}
            {/* {step === 2 && (
              <div className="space-y-6">
                <div className="relative">
                  <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
                  <input
                    type="text" maxLength={6} autoFocus
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                    value={formData.otp}
                    onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
                  />
                </div>
                <button type="button" onClick={() => setStep(3)} className="text-sidebar-primary text-sm font-semibold flex items-center gap-2 hover:underline">
                  <ShieldQuestion size={18} /> Try another way
                 </button>
              </div>
            )} */}

            {/* Step 2: Verification Code */}
            {step === 2 && (
              <div className="space-y-6">
                <div className="relative">
                  <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">Code -</span>
                  <input
                    type="text" maxLength={6} autoFocus
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-18 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                    value={formData.otp}
                    onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
                  />
                </div>

                <div className="flex flex-wrap items-center gap-4">
                  <button
                    type="button"
                    onClick={() => setStep(3)}
                    className="text-muted-foreground text-sm font-semibold flex items-center gap-2 hover:text-sidebar-primary transition-colors"
                  >
                    <ShieldQuestion size={18} /> Try another way
      </button>

                  <div className="h-4 w-[1px] bg-border hidden sm:block" />

                  <button
                    type="button"
                    disabled={!canResend || isLoading}
                    onClick={handleIdentify}
                    className={cn(
                      "text-sm font-semibold transition-all",
                      canResend ? "text-sidebar-primary hover:underline" : "text-muted-foreground cursor-not-allowed"
                    )}
                  >
                    {canResend ? "Resend OTP" : `Resend in ${resendCountdown}s`}
                  </button>
                </div>
              </div>
            )}

            {/* Step 3: Manual Recovery Entry (Blind) */}
            {step === 3 && (
              <div className="relative group">
                <input
                  type="text" required
                  className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 text-foreground outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                  placeholder="Recovery email/phone"
                  value={formData.identifier}
                  onChange={(e) => setFormData({ ...formData, identifier: e.target.value })}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all pointer-events-none peer-focus:-top-3 peer-focus:left-3 peer-focus:text-sm peer-focus:text-sidebar-primary peer-focus:bg-card peer-focus:px-2 peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:left-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card peer-[:not(:placeholder-shown)]:px-2">
                  Recovery email or phone
                </label>
              </div>
            )}

            {/* Step 4: New Password */}
            {step === 4 && (
              <div className="space-y-4">
                <div className="relative">
                  <input
                    type={showPass ? "text" : "password"} required
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                    placeholder="New password"
                    onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                  />
                  <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm px-1">New password</label>
                </div>
                <div className="relative">
                  <input
                    type={showPass ? "text" : "password"} required
                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                    placeholder="Confirm"
                    onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                  />
                  <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm px-1">Confirm</label>
                  <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
                    {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
                  </button>
                </div>
              </div>
            )}

            {step === 5 && (
              <div className="text-center space-y-6 animate-in zoom-in duration-500">
                <div className="flex justify-center"><CheckCircle2 size={64} className="text-[#34A853]" /></div>
                <h2 className="text-2xl font-medium text-foreground">All done!</h2>
                <Link href="/login" className="block w-full bg-sidebar-primary text-white font-semibold py-4 rounded-2xl hover:shadow-lg transition-all text-center">
                  Go to Login Now
                </Link>
              </div>
            )}
            {/* {error && step < 5 && (
              <div className="flex items-center gap-2 text-red-600 text-sm px-2 animate-in fade-in slide-in-from-top-1">
                <AlertCircle size={16} />
                <span>{error}</span>
              </div>
            )} */}
            {error && (
              <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
                <AlertCircle size={16} />
                <span>{error}</span>
              </div>
            )}

            {step < 5 && <div className="flex items-center justify-between pt-8">
              <button type="button"
                onClick={() => router.push("/login")}
                className="text-primary font-bold gap-2 pl-2">Cancel</button>
              <div /> {/* Placeholder for alignment since we removed "Back to Login" */}
              <button type="submit" disabled={isLoading} className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:shadow-lg transition-all flex items-center gap-2">
                {isLoading && <Loader2 size={18} className="animate-spin" />}
                {step === 4 ? "Update Password" : step === 3 ? "Send Code" : "Next"}
              </button>
            </div>}
          </form>
        </div>
      </div>
    </div>
  );
}