"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import {
    ChevronDown, Eye, EyeOff, ArrowLeft, UserCircle2,
    Trash2, PlusCircle, Loader2, AlertCircle, UserMinus
} from "lucide-react";
import { cn } from "@/lib/utils";
import { useSearchParams, useRouter } from 'next/navigation';
import { authService } from "@/lib/auth"
import { useAuth } from "@/context/AuthContext";
import { Logo } from "@/components/ui/Logo"
import PrivacyPolicyFooter from '@/components/layout/PrivacyPolicyFooter'
const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL
import { redirect } from 'next/navigation';

interface SavedAccount {
    email: string;
    name?: string;
    user_id?: string;
    isLoggedIn?: boolean; // True if there's an active session in DB
}

interface ApiAccount {
    user_id: string;
    identifier: string;
    active: boolean;
}

export default function LoginPage() {
    const [step, setStep] = useState(1);
    const [email, setEmail] = useState("");
    const [otp, setOtp] = useState("");
    const [showPass, setShowPass] = useState(false);
    const [mergedAccounts, setMergedAccounts] = useState<SavedAccount[]>([]);
    const [isLoading, setIsLoading] = useState(false);
    const [isFetchingSessions, setIsFetchingSessions] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [timer, setTimer] = useState(0);

    const searchParams = useSearchParams();
    const router = useRouter();
    const authParam = searchParams.get("auth");
    const redirectEmail = searchParams.get("email");
    const { refreshUser } = useAuth();
    const client_id = searchParams.get("client_id") || ""
    const callback_url = searchParams.get("callback_url") || ""

    const authenticateApps = async () => {
        const deviceId = localStorage.getItem("device_id");

        if (client_id && callback_url) {
            console.log(client_id)
            console.log(callback_url)
            try {
                const res = await fetch(`${API_URL}/api/oauth/authorize/?client_id=${client_id}&callback_url=${callback_url}`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Device-Id': deviceId || ""
                    },
                    credentials: "include",
                });
                const data = await res.json()
                console.log(data)
                if(res.ok){
                    window.location.href=data.callback_url
                }else{
                    window.location.href=data.callback_url
                }
            } catch (err) {
                // console.error(err);
                setError("Something Went Wrong!")
                
            } finally {
            }
        }
    }
    // 1. Sync Logic: Combine LocalStorage + Backend Sessions
    const syncAllAccounts = async () => {
        setIsFetchingSessions(true);
        const deviceId = localStorage.getItem("device_id");

        try {
            // Fetch active sessions from DB
            let apiAccounts: ApiAccount[] = [];
            if (deviceId) {
                const res = await fetch(`${API_URL}/api/auth/accounts/`, {
                    headers: { 'X-Device-Id': deviceId }
                });
                if (res.ok) apiAccounts = await res.json();
            }

            // Get local storage history
            const saved = localStorage.getItem("saved_accounts");
            const localAccounts: SavedAccount[] = saved ? JSON.parse(saved) : [];

            // Merge: API accounts take priority (isLoggedIn: true)
            const activeOnes = apiAccounts.map(api => ({
                email: api.identifier,
                name: api.identifier.split('@')[0],
                user_id: api.user_id,
                isLoggedIn: true
            }));

            // Add local accounts that aren't currently "active" sessions
            const inactiveOnes = localAccounts.filter(local =>
                !apiAccounts.some(api => api.identifier === local.email)
            ).map(local => ({ ...local, isLoggedIn: false }));

            const combined = [...activeOnes, ...inactiveOnes];
            setMergedAccounts(combined);

            // Logic to determine initial step
            if (redirectEmail) {
                selectAccount({ email: redirectEmail });
            } else if (authParam && combined[Number(authParam) - 1]) {
                selectAccount(combined[Number(authParam) - 1]);
            } else if (combined.length > 0 && step === 1) {
                setStep(0); // Show picker if accounts exist
            }
        } catch (err) {
            // console.error("Sync failed", err);
            setError("Something Went Wrong!")
        } finally {
            setIsFetchingSessions(false);
        }
    };

    useEffect(() => {
        syncAllAccounts();
    }, [authParam, redirectEmail]);

    // OTP Timer
    useEffect(() => {
        let interval: NodeJS.Timeout;
        if (step === 3 && timer > 0) {
            interval = setInterval(() => setTimer((t) => t - 1), 1000);
        }
        return () => clearInterval(interval);
    }, [step, timer]);

    const handleRemoveAccount = async (e: React.MouseEvent, acc: SavedAccount) => {
        e.stopPropagation();
        const deviceId = localStorage.getItem("device_id");

        try {
            if (acc.isLoggedIn && acc.user_id) {
                // Remove session from DB
                await fetch(`${API_URL}/api/auth/account/delete/`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Device-Id': deviceId || ""
                    },
                    credentials: "include",
                    body: JSON.stringify({ user_id: acc.user_id })
                });
            }

            // Remove from Local Storage
            const saved = localStorage.getItem("saved_accounts");
            if (saved) {
                const updated = JSON.parse(saved).filter((a: any) => a.email !== acc.email);
                localStorage.setItem("saved_accounts", JSON.stringify(updated));
            }

            setMergedAccounts(prev => prev.filter(a => a.email !== acc.email));
            if (mergedAccounts.length <= 1) setStep(1);
        } catch (err) {
            setError("Failed to remove account");
        }
    };


    const handleResendOtp = async () => {
        // if (timer > 0 || isLoading) return;
        if (timer > 0) return;

        setIsLoading(true);
        setTimer(30)
        try {
            const res = await fetch(`${API_URL}/api/auth/otp/send/`, {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ identifier: email }),
            });
            const data = await res.json()
            if (!res.ok) throw new Error(data.error || "Failed to resend code");
            setTimer(30);
        } catch (err: any) {
            setError(err.message || "Something Went Wrong!");
        } finally {
            setIsLoading(false);
        }
    };


    const handleNext = async (e: React.FormEvent) => {
        e.preventDefault();
        setError(null);

        if (step === 1) {
            setStep(2);
            return;
        }

        const passwordInput = (e.currentTarget.querySelector('input[name="password"]') as HTMLInputElement)?.value;

        if (step === 2) {
            setIsLoading(true);
            try {
                const response = await authService.login(email, passwordInput);
                const data = await response.json();

                // Update Local Storage history
                const updated = [
                    { email, name: email.split("@")[0] },
                    ...mergedAccounts.filter(a => a.email !== email)
                ].slice(0, 5);
                localStorage.setItem("saved_accounts", JSON.stringify(updated));
                console.log(data)
                if (response.status === 403 && !data.is_active) {
                    setError(data.error)
                    return;
                }

                if (response.status === 403 && !data.is_verified) {
                    handleResendOtp()
                    setStep(3);
                    setError(data.error || "")
                    return;
                }

                if (response.ok) {
                    await refreshUser();
                    if (client_id && callback_url) {
                        await authenticateApps()
                    } else {
                        router.push('/dashboard');
                    }
                } else {
                    setError(data.error || "Invalid credentials");
                }
            } catch (err) {
                setError("Connection failed");
            } finally {
                setIsLoading(false);
            }
        }

        // STEP 5: FINAL OTP VERIFICATION
        if (step === 3) {
            try {
                setIsLoading(true);
                const res = await fetch(`${API_URL}/api/auth/otp/verify/`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({
                        identifier: email,
                        otp: otp
                    }),
                });
                const data = await res.json();
                if (!res.ok) {
                    setError(data.error || "Invalid credentials");
                }
                // Registration successful
                window.location.href = `/login?email=${email}`;
                return;
            } catch (e) {
                setError("Internal Server error");
            } finally {
                setIsLoading(false);
            }
        }
    };

    const [switchingId, setSwitchingId] = useState<string | null>(null);
    const handleSwitchAccount = async (userId: string, email: string) => {
        // const device_id = validateDevice()
        const deviceId = localStorage.getItem("device_id");
        setSwitchingId(userId);
        try {
            const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/switch-account/`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Device-Id': deviceId || "",
                },
                credentials: "include",
                body: JSON.stringify({ user_id: userId })
            });
            const data = await response.json()
            console.log(response)

            if (response.ok) {
                // router.push("/dashboard")
                if (client_id && callback_url) {
                    await authenticateApps()
                } else {
                    router.push('/dashboard');
                }

                return
            } else if (response.status == 401 || response.status==403) {
                setEmail(email);
                setStep(2);
            }

        } catch (error) {
            // console.error("Failed to switch account", error);
            setError("Failed to switch account")
        } finally {
            setSwitchingId(null);
        }
    };

    const selectAccount = (acc: SavedAccount) => {
        if (acc.isLoggedIn) {
            handleSwitchAccount(acc.user_id, acc.email)
        } else {
            setEmail(acc.email);
            setStep(2);
        }
    };

    const getInitial = (name: string) => name?.charAt(0).toUpperCase() || "?";

    return (

        <div className="w-full bg-background flex flex-col items-center justify-center p-0 sm:p-4 min-h-screen">
            <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] overflow-hidden flex flex-col md:flex-row min-h-[500px] transition-all duration-500 shadow-2xl">

                {/* Left Side: Branding & Dynamic Headers */}
                <div className="w-full md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
                    <div className="space-y-6">
                        <Logo />
                        <div className="space-y-2">
                            <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                                {step === 0 && "Choose an account"}
                                {step === 1 && "Sign in"}
                                {step === 2 && "Welcome back"}
                                {step === 3 && "Verify it's you"}
                            </h1>
                            <p className="text-muted-foreground text-lg font-light max-w-sm">
                                {step === 3 ? `Enter the code sent to ${email}` :
                                    step === 2 ? `Enter your password for ${email}` :
                                        "Continue to your Indzs account"}
                            </p>
                        </div>
                    </div>
                    {step > 0 && (
                        <button onClick={() => { setStep(mergedAccounts.length > 0 ? 0 : 1); setError(null) }} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all -ml-4">
                            <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
                        </button>
                    )}
                </div>

                {/* Right Side: Interactive Forms */}
                <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">

                    {/* STEP 0: Account Picker */}
                    {step === 0 && (
                        <div className="space-y-4 animate-in fade-in slide-in-from-bottom-4">
                            <div className="border border-border rounded-[24px] overflow-hidden divide-y bg-card/50">
                                {isFetchingSessions ? (
                                    /* Skeleton Loading */
                                    [1, 2].map((i) => (
                                        <div key={i} className="flex items-center gap-3 p-4 animate-pulse">
                                            <div className="w-10 h-10 bg-muted rounded-full" />
                                            <div className="flex-1 space-y-2">
                                                <div className="h-3 bg-muted rounded w-1/3" />
                                                <div className="h-2 bg-muted rounded w-1/2" />
                                            </div>
                                        </div>
                                    ))
                                ) : (
                                        mergedAccounts.map((acc) => (
                                            <div key={acc.email} onClick={() => selectAccount(acc)} className="flex items-center justify-between p-4 hover:bg-accent cursor-pointer group transition-colors">
                                                <div className="flex items-center gap-3">
                                                    {/* <div className="w-10 h-10 rounded-full bg-sidebar-primary/10 flex items-center justify-center text-sidebar-primary">
                                                        <UserCircle2 size={24} />
                                                    </div> */}
                                                    <div className="w-9 h-9 rounded-full bg-muted flex items-center justify-center text-muted-foreground text-sm font-bold overflow-hidden">
                                                        {switchingId === acc.user_id ? (
                                                            <Loader2 className="h-4 w-4 animate-spin" />
                                                        ) : (
                                                                acc.image ? <img src={acc.image} alt="" /> : getInitial(acc.email)
                                                            )}
                                                    </div>
                                                    <div className="flex flex-col">
                                                        <span className="text-sm font-medium flex items-center gap-2">
                                                            {acc.name || acc.email.split('@')[0]}
                                                            {acc.isLoggedIn && <span className="w-2 h-2 bg-green-500 rounded-full" title="Active session" />}
                                                        </span>
                                                        <span className="text-xs text-muted-foreground break-all">{acc.email}</span>
                                                    </div>
                                                </div>
                                                <div className="flex items-center gap-2">
                                                    {!acc.isLoggedIn && <span className="text-[10px] bg-muted px-1.5 py-0.5 rounded text-muted-foreground">Signed out</span>}

                                                    <button onClick={(e) => handleRemoveAccount(e, acc)} className="p-2 md:opacity-0 group-hover:opacity-100 text-muted-foreground hover:text-red-600 transition-all">
                                                        {acc.isLoggedIn ? <UserMinus size={18} /> : <Trash2 size={18} />}
                                                    </button>
                                                </div>
                                            </div>
                                        ))
                                    )}
                                <div onClick={() => setStep(1)} className="flex items-center gap-3 p-4 hover:bg-accent cursor-pointer text-sidebar-primary">
                                    <PlusCircle size={20} />
                                    <span className="text-sm font-medium">Use another account</span>
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Login Forms */}
                    {step >= 1 && (
                        <form onSubmit={handleNext} className="space-y-6">
                            {/* Email Step */}
                            {step === 1 && (
                                <div className="space-y-4">
                                    <div className="relative">
                                        <input
                                            type="" required autoFocus
                                            className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none transition-all"
                                            placeholder="Email or Phone" value={email} onChange={(e) => setEmail(e.target.value)}
                                        />
                                        <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Email address</label>
                                    </div>
                                    <Link href="/recover-account" className="text-sidebar-primary text-sm font-semibold hover:underline block">Forgot Email?</Link>
                                </div>
                            )}

                            {/* Password Step */}
                            {step === 2 && (
                                <div className="space-y-4">
                                    <div className="relative">
                                        <input
                                            name="password"
                                            type={showPass ? "text" : "password"} required autoFocus
                                            className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none transition-all"
                                            placeholder="Password"
                                        />
                                        <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Enter password</label>
                                        <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
                                            {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
                                        </button>
                                    </div>
                                    <Link href="/forgot-password" size="sm" className="text-sidebar-primary text-sm font-semibold hover:underline block">Forgot password?</Link>
                                </div>
                            )}

                            {/* OTP Step */}
                            {step === 3 && (
                                <div className="space-y-4">
                                    <div className="relative">
                                        <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">Code -</span>
                                        <input
                                            type="text" maxLength={6} required autoFocus
                                            className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-18 pr-5 py-4 tracking-[0.5em] font-bold outline-none"
                                            value={otp} onChange={(e) => setOtp(e.target.value.replace(/\D/g, ""))}
                                        />
                                    </div>
                                    <button type="button" onClick={() => setTimer(30)} disabled={timer > 0} className="text-sm text-sidebar-primary disabled:text-muted-foreground font-medium">
                                        Resend code {timer > 0 && `(${timer}s)`}
                                    </button>
                                </div>
                            )}

                            {error && (
                                <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
                                    <AlertCircle size={16} />
                                    <span>{error}</span>
                                </div>
                            )}

                            <div className="flex items-center justify-between pt-4">
                                <Link href="/register" className="text-sidebar-primary font-semibold text-sm">Create account</Link>
                                <button
                                    type="submit" disabled={isLoading}
                                    className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:shadow-lg hover:shadow-sidebar-primary/30 disabled:opacity-50 flex items-center gap-2 transition-all"
                                >
                                    {isLoading && <Loader2 className="w-4 h-4 animate-spin" />}
                                    {step === 3 ? "Verify" : "Next"}
                                </button>
                            </div>
                        </form>
                    )}
                </div>
            </div>
        </div>
    );
}

// "use client";

// import { useState, useEffect } from "react";
// import Link from "next/link";
// import { ChevronDown, Eye, EyeOff, ArrowLeft, UserCircle2, Trash2, PlusCircle, Loader2, AlertCircle } from "lucide-react";
// import { cn } from "@/lib/utils";
// import { useSearchParams, useRouter } from 'next/navigation';
// import { authService } from "@/lib/auth"
// import { useAuth } from "@/context/AuthContext";
// import {Logo} from "@/components/ui/Logo"
// const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL
// interface SavedAccount {
//     email: string;
//     name?: string;

// }

// export default function LoginPage() {
//     // 0: Picker, 1: Email, 2: Password, 3: OTP (for unverified users)
//     const [step, setStep] = useState(1);
//     const [email, setEmail] = useState("");
//     const [otp, setOtp] = useState("");
//     const [showPass, setShowPass] = useState(false);
//     const [savedAccounts, setSavedAccounts] = useState<SavedAccount[]>([]);
//     const [isLoading, setIsLoading] = useState(false);
//     const [error, setError] = useState<string | null>(null);
//     const [timer, setTimer] = useState(30);

//     const searchParams = useSearchParams();
//     const router = useRouter();
//     const auth = searchParams.get("auth") ?? "";
//     const redirectEmail = searchParams.get("email")?? ""
//     // const baseUrl = process.env.NEXT_PUBLIC_API_BASE_URL;

//     const { refreshUser, user, loading } = useAuth();
//     // Timer for OTP
//     useEffect(() => {
//         let interval: NodeJS.Timeout;
//         if (step === 3 && timer > 0) {
//             interval = setInterval(() => setTimer((t) => t - 1), 1000);
//         }
//         return () => clearInterval(interval);
//     }, [step, timer]);

//     useEffect(() => {
//         const accounts = localStorage.getItem("saved_accounts");
//         if (accounts) {
//             const parsed = JSON.parse(accounts);
//             setSavedAccounts(parsed);
//             if (auth && parsed[Number(auth) - 1]) {
//                 selectAccount(parsed[Number(auth) - 1]);
//             } else if (parsed.length > 0) {
//                 setStep(0);
//             }
//         }
//         if(redirectEmail){
//             selectAccount({email: redirectEmail})
//         }
//     }, [auth, redirectEmail]);

//     const getDeviceInfo = () => `Platform: ${navigator.platform}, UA: ${navigator.userAgent}`;

//     const getOrCreateDeviceId = () => {
//         let id = localStorage.getItem("device_id");
//         if (!id) {
//             id = crypto.randomUUID?.() || Math.random().toString(36).substring(2);
//             localStorage.setItem("device_id", id);
//         }
//         return id;
//     };

//     // --- NEW: FORGOT PASSWORD FETCH ---
//     const handleForgotPassword = async () => {
//         // if (!email) {
//         //     setError("Please enter your email first");
//         //     setStep(1);
//         //     return;
//         // }
//         // setIsLoading(true);
//         // try {
//         //     const res = await fetch(`${API_URL}/api/auth/forgot-password/`, {
//         //         method: "POST",
//         //         headers: { "Content-Type": "application/json" },
//         //         body: JSON.stringify({ email }),
//         //     });
//         //     if (res.ok) alert("Password reset link sent to your email.");
//         //     else setError("Failed to send reset link.");
//         // } catch (err) {
//         //     setError("Network error.");
//         // } finally {
//         //     setIsLoading(false);
//         // }
//         router.push("/forgot-password")
//     };

//     // --- NEW: RESEND OTP ---
//     const handleResendOtp = async () => {
//         if (timer > 0 || isLoading) return;
//         setIsLoading(true);
//         try {
//             await fetch(`${API_URL}/api/auth/otp/send/`, {
//                 method: "POST",
//                 headers: { "Content-Type": "application/json" },
//                 body: JSON.stringify({ identifier: email }),
//             });
//             setTimer(30);
//         } catch (err) {
//             setError("Failed to resend.");
//         } finally {
//             setIsLoading(false);
//         }
//     };

//     const handleNext = async (e: React.FormEvent) => {
//         e.preventDefault();
//         setError(null);

//         if (step === 1 && email) {
//             setStep(2);
//             return;
//         }

//         const passwordInput = (e.currentTarget.querySelector('input[type="password"], input[name="password"]') as HTMLInputElement)?.value;

//         // STEP 2: LOGIN ATTEMPT
//         if (step === 2) {
//             setIsLoading(true);
//             try {
//                 // const response = await fetch(`${API_URL}/api/auth/login/`, {
//                 //     method: "POST",
//                 //     headers: {
//                 //         "Content-Type": "application/json",
//                 //         "X-Device-Info": getDeviceInfo(),
//                 //         "X-Device-Id": getOrCreateDeviceId(),
//                 //     },
//                 //     body: JSON.stringify({ identifier: email, password: passwordInput }),
//                 // });

//                 const response = await authService.login(email, passwordInput);
//                 const data = await response.json();
//                 // const data = await response.json();

//                 if (response.status === 403 && !data.is_verified) {
//                     // Logic: If user is not verified, send OTP and move to OTP step
//                     handleResendOtp()
//                     setStep(3);
//                     setTimer(30);
//                     setError("Account Not Verified!")
//                     return;
//                 }

//                 if (response.ok) {
//                     const updatedAccounts = [
//                         { email, name: email.split("@")[0] },
//                         ...savedAccounts.filter(a => a.email !== email)
//                     ].slice(0, 5);
//                     localStorage.setItem("saved_accounts", JSON.stringify(updatedAccounts));
//                     await refreshUser();
//                     router.push('/dashboard');
//                 } else {
//                     setError(data.error || "Invalid credentials");
//                 }
//             } catch (err) {
//                 setError("Connection failed");
//             } finally {
//                 setIsLoading(false);
//             }
//         }

//         // STEP 3: OTP VERIFICATION (Same as Register)
//         if (step === 3) {
//             setIsLoading(true);
//             try {
//                 const res = await fetch(`${API_URL}/api/auth/otp/verify/`, {
//                     method: "POST",
//                     headers: { "Content-Type": "application/json" },
//                     body: JSON.stringify({ identifier: email, otp }),
//                 });
//                 if (res.ok) router.push('/dashboard');
//                 else setError("Invalid OTP code");
//             } catch (err) {
//                 setError("Verification failed");
//             } finally {
//                 setIsLoading(false);
//             }
//         }
//     };

//     const selectAccount = (acc: SavedAccount) => {
//         setEmail(acc.email);
//         setStep(2);
//     };

//     const removeAccount = (e: React.MouseEvent, accEmail: string) => {
//         e.stopPropagation();
//         const updated = savedAccounts.filter(a => a.email !== accEmail);
//         setSavedAccounts(updated);
//         localStorage.setItem("saved_accounts", JSON.stringify(updated));
//         if (updated.length === 0) setStep(1);
//     };

//     return (
//         <div className="w-full bg-background flex flex-col items-center justify-center p-0 sm:p-4 min-h-screen">
//             <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] overflow-hidden flex flex-col md:flex-row min-h-[500px] transition-all duration-500">

//                 {/* Left Side */}
//                 <div className="w-full md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
//                     <div className="space-y-6">
//                         <Logo/>
//                         <div className="space-y-2">
//                             <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
//                                 {step === 0 && "Choose an account"}
//                                 {step === 1 && "Sign in"}
//                                 {step === 2 && "Welcome back"}
//                                 {step === 3 && "Verify it's you"}
//                             </h1>
//                             <p className="text-muted-foreground text-lg font-light max-w-sm">
//                                 {step === 3 ? `Enter the code sent to ${email}` : `Continue to Indzs Account as ${email}`}
//                             </p>
//                         </div>
//                     </div>
//                     {step > 0 && (
//                         <button onClick={() => { setStep(step === 3 ? 2 : 0); setError(null) }} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all -ml-4">
//                             <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
//                         </button>
//                     )}
//                 </div>

//                 {/* Right Side */}
//                 <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">

//                     {step === 0 && (
//                         <div className="space-y-2 animate-in fade-in slide-in-from-bottom-4">
//                             <div className="border border-border rounded-2xl overflow-hidden divide-y">
//                                 {savedAccounts.map((acc) => (
//                                     <div key={acc.email} onClick={() => selectAccount(acc)} className="flex items-center justify-between p-4 hover:bg-accent cursor-pointer group">
//                                         <div className="flex items-center gap-3">
//                                             <UserCircle2 className="text-sidebar-primary" size={24} />
//                                             <div className="flex flex-col">
//                                                 <span className="text-sm font-medium">{acc.name}</span>
//                                                 <span className="text-xs text-muted-foreground">{acc.email}</span>
//                                             </div>
//                                         </div>
//                                         <button onClick={(e) => removeAccount(e, acc.email)} className="p-2 opacity-0 group-hover:opacity-100 text-muted-foreground hover:text-destructive">
//                                             <Trash2 size={16} />
//                                         </button>
//                                     </div>
//                                 ))}
//                                 <div onClick={() => setStep(1)} className="flex items-center gap-3 p-4 hover:bg-accent cursor-pointer">
//                                     <PlusCircle size={24} className="text-muted-foreground" />
//                                     <span className="text-sm font-medium">Use another account</span>
//                                 </div>
//                             </div>
//                         </div>
//                     )}

//                     {(step >= 1) && (
//                         <form onSubmit={handleNext} className="space-y-6">
//                             {step === 1 && (
//                                 <div>
//                                     <div className="relative">
//                                         <input
//                                             type="text" required autoFocus
//                                             className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none"
//                                             placeholder="Email" value={email} onChange={(e) => setEmail(e.target.value)}
//                                         />
//                                         <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Email or phone</label>
//                                     </div>
//                                     <div className="mt-2">

//                                     <Link href="/recover-account" className="text-sidebar-primary text-sm font-semibold hover:underline mt-6"> Forgot Email or Phone?</Link>
//                                     </div>

//                                 </div>
//                             )}

//                             {step === 2 && (
//                                 <div className="relative">
//                                     <input
//                                         name="password"
//                                         type={showPass ? "text" : "password"} required autoFocus
//                                         className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none"
//                                         placeholder="Password"
//                                     />
//                                     <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Enter your password</label>
//                                     <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
//                                         {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
//                                     </button>
//                                 </div>
//                             )}

//                             {step === 3 && (
//                                 <div className="space-y-4">
//                                     <div className="relative">
//                                         <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">G -</span>
//                                         <input
//                                             type="text" maxLength={6} required autoFocus
//                                             className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-12 pr-5 py-4 tracking-[0.5em] font-bold outline-none"
//                                             value={otp} onChange={(e) => setOtp(e.target.value.replace(/\D/g, ""))}
//                                         />
//                                     </div>
//                                     <button type="button" onClick={handleResendOtp} disabled={timer > 0 || isLoading} className="text-sm text-sidebar-primary disabled:text-muted-foreground">
//                                         Resend code {timer > 0 && `(${timer}s)`}
//                                     </button>
//                                 </div>
//                             )}

//                             {step === 2 && (
//                                 <button type="button" onClick={handleForgotPassword} className="text-sidebar-primary text-sm font-semibold hover:underline">
//                                     Forgot password?
//                                 </button>
//                             )}

//                             {error && (
//                                 <div className="flex items-center gap-2 text-red-600 text-sm px-2 animate-in fade-in slide-in-from-top-1" style={{ alignItems: 'center' }}>
//                                     <AlertCircle size={16} />
//                                     <span>{error}</span>
//                                 </div>
//                             )}

//                             <div className="flex items-center justify-between pt-4">
//                                 <Link href="/register" className="text-sidebar-primary font-semibold text-sm">Create account</Link>
//                                 <button
//                                     type="submit" disabled={isLoading}
//                                     className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:opacity-90 flex items-center gap-2"
//                                 >
//                                     {isLoading && <Loader2 className="w-4 h-4 animate-spin" />}
//                                     {step === 3 ? "Verify" : "Next"}
//                                 </button>
//                             </div>
//                         </form>
//                     )}
//                 </div>
//             </div>
//         </div>
//     );
// }