"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { ArrowLeft, Loader2, AlertCircle, UserCircle2, CheckCircle2, ChevronRight } from "lucide-react";
import { cn } from "@/lib/utils";
import { Logo } from "@/components/ui/Logo"
const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL;

interface Account {
  masked_email: string;
  name: string;
  masked_phone?: string
}
export default function RecoverAccountPage() {
  // 1: Identifier (Email/Phone), 2: Name details, 3: OTP, 4: Result
  const [step, setStep] = useState(1);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [timer, setTimer] = useState(30);
  const [recoveryToken, setRecoveryToken] = useState(null)

  // Form State
  const [identifier, setIdentifier] = useState("");
  const [firstName, setFirstName] = useState("");
  const [lastName, setLastName] = useState("");
  const [otp, setOtp] = useState("");
  // const [foundAccount, setFoundAccount] = useState<{email: string, name: string} | null>(null);
  const [foundAccounts, setFoundAccounts] = useState<Account[]>([]);

  const router = useRouter();

  // OTP Timer Logic
  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (step === 3 && timer > 0) {
      interval = setInterval(() => setTimer((t) => t - 1), 1000);
    }
    return () => clearInterval(interval);
  }, [step, timer]);

  const handleSendOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setError(null);

    if (!firstName.trim()) {
      setError("First name is required");
      return;
    }

    setIsLoading(true);
    try {
      const res = await fetch(`${API_URL}/api/auth/find-account/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          identifier,
          first_name: firstName,
          last_name: lastName
        }),
      });
      const data = await res.json();
      console.log(data)
      if (res.ok) {
        setRecoveryToken(data.recovery_token)
        setStep(3);
        setTimer(30);
      } else {
        setError(data.error || "No account found with these details.");
      }
    } catch (err) {
      setError("Connection failed. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  const handleVerifyOTP = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);
    setError(null);

    try {
      const res = await fetch(`${API_URL}/api/auth/find-account/verify-otp/`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ recovery_token: recoveryToken, otp: otp }),
      });
      const data = await res.json();
      console.log(data)
      if (res.ok) {
        // setFoundAccount({ email: data.masked_email, name: data.name });
        setFoundAccounts(data.accounts || []);
        setStep(4);
      } else {
        setError(data.error || "Invalid OTP code.");
      }
    } catch (err) {
      setError("Verification failed.");
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="w-full bg-background flex flex-col items-center justify-center p-0 sm:p-4 min-h-screen">
      <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] overflow-hidden flex flex-col md:flex-row min-h-[500px] transition-all duration-500">

        {/* Left Side: Dynamic Messaging */}
        <div className="w-full md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
          <div className="space-y-6">
            <Logo />
            <div className="space-y-2">
              <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                {step === 1 && "Find your Account"}
                {step === 2 && "What's your name?"}
                {step === 3 && "Verify it's you"}
                {step === 4 && "Account found"}
              </h1>
              <p className="text-muted-foreground text-lg font-light max-w-sm">
                {step === 1 && "Enter your phone number or recovery email"}
                {step === 2 && "Enter the name on your Indzs Account"}
                {step === 3 && `Enter the code sent to ${identifier}`}
                {step === 4 && "You can now proceed to sign in"}
              </p>
            </div>
          </div>
          {step > 1 && step < 4 && (
            <button onClick={() => setStep(step - 1)} className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all -ml-4">
              <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
            </button>
          )}
        </div>

        {/* Right Side: Forms */}
        <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">

          {/* Step 1: Identifier */}
          {step === 1 && (
            <form onSubmit={(e) => { e.preventDefault(); setStep(2); }} className="space-y-6">
              <div className="relative">
                <input
                  type="text" required autoFocus
                  className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none"
                  placeholder="Email or phone" value={identifier} onChange={(e) => setIdentifier(e.target.value)}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Email or phone</label>
              </div>
              <div className="flex justify-between pt-4" style={{ alignItems: "center" }}>
                <Link href="/login" className="text-sidebar-primary text-sm font-semibold hover:underline block">
                  Back to Login
                </Link>
                <button type="submit" className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:opacity-90">Next</button>
              </div>
            </form>
          )}

          {/* Step 2: Name Validation */}
          {step === 2 && (
            <form onSubmit={handleSendOTP} className="space-y-6">
              <div className="relative">
                <input
                  type="text" required autoFocus
                  className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none"
                  placeholder="First Name" value={firstName} onChange={(e) => setFirstName(e.target.value)}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">First name</label>
              </div>
              <div className="relative">
                <input
                  type="text"
                  className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 peer placeholder-transparent outline-none"
                  placeholder="Last Name" value={lastName} onChange={(e) => setLastName(e.target.value)}
                />
                <label className="absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Last name (optional)</label>
              </div>
              {/* {error && <div className="flex items-center gap-2 text-red-600 text-sm animate-in fade-in"><AlertCircle size={16} />{error}</div>} */}
              {error && (
                <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
                  <AlertCircle size={16} />
                  <span>{error}</span>
                </div>
              )}
              <div className="flex justify-end pt-4">
                <button type="submit" disabled={isLoading} className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl flex items-center gap-2">
                  {isLoading && <Loader2 className="w-4 h-4 animate-spin" />} Next
                                </button>
              </div>
            </form>
          )}

          {/* Step 3: OTP */}
          {step === 3 && (
            <form onSubmit={handleVerifyOTP} className="space-y-6">
              <div className="relative">
                <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">Code -</span>
                <input
                  type="text" maxLength={6} required autoFocus
                  className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-18 pr-5 py-4 tracking-[0.5em] font-bold outline-none"
                  value={otp} onChange={(e) => setOtp(e.target.value.replace(/\D/g, ""))}
                />
              </div>
              {error && <div className="flex items-center gap-2 text-red-600 text-sm"><AlertCircle size={16} />{error}</div>}
              <div className="flex items-center justify-between pt-4">
                <button type="button" onClick={handleSendOTP} disabled={timer > 0} className="text-sm text-sidebar-primary disabled:text-muted-foreground">
                  Resend {timer > 0 && `(${timer}s)`}
                </button>
                <button type="submit" disabled={isLoading} className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl flex items-center gap-2">
                  {isLoading && <Loader2 className="w-4 h-4 animate-spin" />} Verify
                                </button>
              </div>
            </form>
          )}

          {/* Step 4: Success / Display Account */}

          {/* Step 4: Account List */}
          {step === 4 && (
            <div className="space-y-4 animate-in fade-in slide-in-from-bottom-4">
              <div className="border border-border rounded-2xl overflow-hidden divide-y divide-border/50">
                {foundAccounts.map((acc, index) => (
                  <div
                    key={acc.masked_email || acc.masked_phone}
                    onClick={() => router.push(`/login?email=${acc.masked_email || acc.masked_phone}`)}
                    className="flex items-center justify-between p-4 hover:bg-sidebar-primary/5 cursor-pointer group transition-colors"
                  >
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 rounded-full bg-sidebar-primary/10 flex items-center justify-center">
                        <UserCircle2 className="text-sidebar-primary" size={24} />
                      </div>
                      <div className="flex flex-col">
                        <span className="text-sm font-semibold text-foreground">{acc.name}</span>
                        <span className="text-xs text-muted-foreground">{acc.masked_email || acc.masked_phone}</span>
                      </div>
                    </div>
                    <ChevronRight size={18} className="text-muted-foreground group-hover:translate-x-1 transition-transform" />
                  </div>
                ))}
              </div>

              <Link href="/login" className="block text-center text-sm text-sidebar-primary font-medium hover:underline pt-2">
                Use a different account
                            </Link>
            </div>
          )}

          {/* {step === 4 && foundAccount && (
                        <div className="space-y-6 animate-in fade-in zoom-in-95">
                            <div className="flex items-center gap-4 p-4 border border-sidebar-primary/20 bg-sidebar-primary/5 rounded-2xl">
                                <UserCircle2 className="text-sidebar-primary w-12 h-12" />
                                <div>
                                    <h3 className="font-semibold text-foreground">{foundAccount.name}</h3>
                                    <p className="text-sm text-muted-foreground">{foundAccount.email}</p>
                                </div>
                                <CheckCircle2 className="ml-auto text-green-500" size={20} />
                            </div>
                            <button 
                                onClick={() => router.push(`/login?email=${foundAccount.email}`)}
                                className="w-full bg-sidebar-primary text-white font-semibold py-4 rounded-2xl hover:opacity-90"
                            >
                                Sign in now
                            </button>
                        </div>
                    )} */}
        </div>
      </div>
    </div>
  );
}