"use client";
import { useState, useEffect } from "react";
import Link from "next/link";
import { ArrowLeft, Eye, EyeOff, AlertCircle, Loader2 } from "lucide-react";
import { cn } from "@/lib/utils";
import { validateStep } from "@/lib/auth-validation";
import { Logo } from "@/components/ui/Logo"
import CountrySelector from "@/components/CountrySelector";
import { Country } from "@/lib/country-data";
const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL
export default function RegisterPage() {
    const [step, setStep] = useState(1);
    const [timer, setTimer] = useState(0);
    const [showPass, setShowPass] = useState(false);
    const [error, setError] = useState("");
    const [isLoading, setIsLoading] = useState(false);

    const [formData, setFormData] = useState({
        firstName: "",
        lastName: "",
        phone: "",
        email: "",
        password: "",
        confirmPassword: "",
        otp: "",
        countryCode:"91"
    });

    useEffect(() => { setError(""); }, [step]);

    // Timer logic for the final OTP step
    useEffect(() => {
        let interval: NodeJS.Timeout;
        if (step === 5 && timer > 0) {
            interval = setInterval(() => setTimer((t) => t - 1), 1000);
        }
        return () => clearInterval(interval);
    }, [step, timer]);

    const handleResendOtp = async () => {
        // if (timer > 0 || isLoading) return;
        if (timer > 0) return;

        setIsLoading(true);
        setTimer(30)
        try {
            const res = await fetch(`${API_URL}/api/auth/otp/send/`, {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ identifier: formData.phone }),
            });
            const data = await res.json()
            console.log(data)
            if (!res.ok) throw new Error(data.error || "Failed to resend code");
            setTimer(30);
        } catch (err: any) {
            setError(err.message || "Something Went Wrong!");
        } finally {
            setIsLoading(false);
        }
    };

    const handleNext = async (e: React.FormEvent) => {
        e.preventDefault();

        const validationError = validateStep(step, formData);
        if (validationError) {
            setError(validationError);
            return;
        }
        setError("");

        try {
            // STEP 4: User has filled Name, Phone, Email, and Password.
            // We save the data now and trigger the OTP for the final step.
            if (step === 4) {
                setIsLoading(true);
                const res = await fetch(`${API_URL}/api/auth/register/`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({
                        first_name: formData.firstName,
                        last_name: formData.lastName,
                        phone_number: formData.phone,
                        primary_email: formData.email,
                        password: formData.password,
                        country_code : formData.countryCode
                    }),
                });
                const data = await res.json();
                if (!res.ok) throw new Error(data.error || "Failed to save data");
                if (res.ok) {
                    // setIsLoading(false)
                    handleResendOtp()
                    // setTimer(30);
                    setStep(5);

                }
                return;
            }

            // STEP 5: FINAL OTP VERIFICATION
            if (step === 5) {
                setIsLoading(true);
                console.log(formData.phone)
                const res = await fetch(`${API_URL}/api/auth/otp/verify/`, {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({
                        identifier: formData.phone,
                        otp: formData.otp
                    }),
                });
                const data = await res.json();
                if (!res.ok) throw new Error(data.error || "Invalid OTP");

                // Registration successful
                window.location.href = `/login?email=${formData.email}`;
                return;
            }

            // Normal progression for Steps 1, 2, 3
            setStep(step + 1);

        } catch (err: any) {
            setError(err.message || "Something Went Wrong!");
        } finally {
            setIsLoading(false);
        }
    };

    const handleCountryChange = (country: Country) => {
        console.log("Selected:", country.name, country.dial);
        setFormData({...formData,countryCode:country.dial })
    };

    return (
        <div className="min-h-screen bg-background flex flex-col md:items-center md:justify-center p-0 sm:p-4 w-full">
            <div className="border-0 sm:border sm:border-border w-full max-w-[1024px] bg-card/80 backdrop-blur-xl rounded-[32px] overflow-hidden flex flex-col md:flex-row min-h-[550px] transition-all duration-500">

                {/* Left Side: Dynamic Branding */}
                <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-between bg-background sm:bg-gradient-to-br sm:from-sidebar/50 sm:to-transparent">
                    <div className="space-y-6">
                        <Logo />
                        <div className="space-y-2">
                            <h1 className="text-3xl md:text-5xl font-medium tracking-tight text-foreground">
                                {step === 1 && "Create account"}
                                {step === 2 && "Your phone"}
                                {step === 3 && "Email address"}
                                {step === 4 && "Safe password"}
                                {step === 5 && "Verify OTP"}
                            </h1>
                            <p className="text-muted-foreground text-lg font-light max-w-sm">
                                {step === 5 ? `Code sent to ${formData.phone}` : "Join our community today"}
                            </p>
                        </div>
                    </div>
                    {step > 1 && (
                        <button
                            type="button"
                            onClick={() => setStep(step - 1)}
                            disabled={isLoading}
                            className="flex items-center gap-2 text-sidebar-primary hover:bg-sidebar-primary/10 w-fit px-4 py-2 rounded-full transition-all -ml-4"
                        >
                            <ArrowLeft size={18} /> <span className="text-sm font-medium">Back</span>
                        </button>
                    )}
                </div>

                {/* Right Side: Form */}
                <div className="md:flex-1 p-8 md:p-12 flex flex-col justify-center bg-background/40">
                    <form onSubmit={handleNext} className="space-y-6">

                        {/* STEP 1: NAME */}
                        {step === 1 && (
                            <div className="space-y-4">
                                <div className="relative">
                                    <input
                                        type="text"
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none transition-all peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="First name"
                                        value={formData.firstName}
                                        onChange={(e) => setFormData({ ...formData, firstName: e.target.value })}
                                    />
                                    <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">First name</label>
                                </div>
                                <div className="relative">
                                    <input
                                        type="text"
                                        className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none transition-all peer placeholder-transparent"
                                        placeholder="Last name"
                                        value={formData.lastName}
                                        onChange={(e) => setFormData({ ...formData, lastName: e.target.value })}
                                    />
                                    <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Last name (optional)</label>
                                </div>
                            </div>
                        )}

                        {/* STEP 2: PHONE */}
                        {step === 2 && (
                            <div className="relative">
                                <div className="absolute left-4 top-1/2 -translate-y-1/2 flex items-center gap-2 border-r border-border pr-2 text-muted-foreground">
                                    {/* <img src="https://flagcdn.com/w20/in.png" width="20" alt="IN" />
                                    <span className="text-sm">+91</span> */}
                                    <CountrySelector onChange={handleCountryChange} defaultValue="IN" />
                                </div>
                                <input
                                    type="tel"
                                    className={cn("w-full  bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-[140px] pr-5 py-4 outline-none", error && "border-destructive")}
                                    placeholder="Phone number"
                                    value={formData.phone}
                                    onChange={(e) => setFormData({ ...formData, phone: e.target.value.replace(/\D/g, "") })}
                                />
                            </div>
                        )}

                        {/* STEP 3: EMAIL */}
                        {step === 3 && (
                            <div className="relative">
                                <input
                                    type="email"
                                    className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                    placeholder="Email"
                                    value={formData.email}
                                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                                />
                                <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Email address</label>
                            </div>
                        )}

                        {/* STEP 4: PASSWORD */}
                        {step === 4 && (
                            <div className="space-y-4">
                                <div className="relative">
                                    <input
                                        type={showPass ? "text" : "password"}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="Password"
                                        value={formData.password}
                                        onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                                    />
                                    <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Password</label>
                                </div>
                                <div className="relative">
                                    <input
                                        type={showPass ? "text" : "password"}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl px-5 py-4 outline-none peer placeholder-transparent", error && "border-destructive")}
                                        placeholder="Confirm"
                                        value={formData.confirmPassword}
                                        onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                                    />
                                    <label className="pointer-events-none absolute left-5 top-4 text-muted-foreground transition-all peer-focus:-top-3 peer-focus:text-sm peer-focus:bg-card peer-[:not(:placeholder-shown)]:-top-3 peer-[:not(:placeholder-shown)]:text-sm peer-[:not(:placeholder-shown)]:bg-card px-1">Confirm</label>
                                    <button type="button" onClick={() => setShowPass(!showPass)} className="absolute right-5 top-4 text-muted-foreground">
                                        {showPass ? <EyeOff size={20} /> : <Eye size={20} />}
                                    </button>
                                </div>
                            </div>
                        )}

                        {/* STEP 5: OTP */}
                        {step === 5 && (
                            <div className="space-y-2">
                                <div className="relative">
                                    <span className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground font-medium">Code -</span>
                                    <input
                                        type="text"
                                        maxLength={6}
                                        className={cn("w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-2xl pl-18 pr-5 py-4 text-foreground tracking-[0.5em] font-bold outline-none", error && "border-destructive")}
                                        value={formData.otp}
                                        onChange={(e) => setFormData({ ...formData, otp: e.target.value.replace(/\D/g, "") })}
                                    />
                                </div>
                                <button type="button" onClick={handleResendOtp} disabled={timer > 0 || isLoading} className="text-sm text-sidebar-primary disabled:text-muted-foreground">
                                    Resend code {timer > 0 && `(${timer}s)`}
                                </button>
                            </div>
                        )}

                        {/* {error && (
                            <div className="flex items-center gap-2 text-red-600 text-sm px-2 animate-in fade-in slide-in-from-top-1">
                                <AlertCircle size={16} />
                                <span>{error}</span>
                            </div>
                        )} */}
                        {error && (
                            <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
                                <AlertCircle size={16} />
                                <span>{error}</span>
                            </div>
                        )}

                        <div className="flex items-center justify-between pt-4">
                            <Link href="/login" className="text-sidebar-primary font-semibold text-sm hover:underline">Sign in instead</Link>
                            <button
                                type="submit"
                                disabled={isLoading}
                                className="bg-sidebar-primary text-white font-semibold px-10 py-3 rounded-2xl hover:opacity-90 transition-all flex items-center gap-2 disabled:opacity-70"
                            >
                                {isLoading && <Loader2 className="h-4 w-4 animate-spin" />}
                                {step === 4 ? "Register" : step === 5 ? "Verify" : "Next"}
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}