"use client";
import React, { useState, useMemo } from "react";
import {
    Search, ChevronDown, Send, MessageSquare,
    Loader2, ArrowRight, HelpCircle,
    X, CheckCircle2, Copy, Check, ArrowLeft, LifeBuoy
} from "lucide-react";
import { cn } from "@/lib/utils";
import { Toast } from '@/components/ui/Toast';
import { BasicNavbar } from '@/components/layout/BasicNavbar'
// const faqs = [
//     { id: 1, question: "How do I manage administrative roles?", answer: "Go to User Management, select a user, and toggle 'Grant Administrator Privileges'." },
//     { id: 2, question: "What happens when I suspend a user?", answer: "Suspended users are logged out and blocked from logging back in until reactivated." },
//     { id: 3, question: "How can I export user data?", answer: "Data export is currently restricted to CSV format for Super Admins only." },
//     { id: 4, question: "How do I restore a deleted account?", answer: "Use the 'Deleted' filter in User Management and click the 'Restore' icon." }
// ];

const faqs = [
    {
        id: 1,
        question: "How do I manage administrative roles?",
        answer: "Go to User Management, select a user, and toggle 'Grant Administrator Privileges'.",
        keywords: ["admin", "permissions", "privileges", "access", "role"]
    },
    {
        id: 2,
        question: "What happens when I suspend a user?",
        answer: "Suspended users are logged out and blocked from logging back in until reactivated.",
        keywords: ["block", "ban", "deactivate", "suspend", "disable"]
    },
    {
        id: 3,
        question: "How can I export user data?",
        answer: "Data export is currently restricted to CSV format for Super Admins only.",
        keywords: ["download", "csv", "excel", "report", "extract"]
    },
    {
        id: 4,
        question: "How do I restore a deleted account?",
        answer: "Use the 'Deleted' filter in User Management and click the 'Restore' icon.",
        keywords: ["undo", "trash", "recovery", "reactivate", "hidden"]
    }
];

const API_URL = process.env.NEXT_PUBLIC_API_BASE_URL

export default function HelpPage() {
    const [step, setStep] = useState<"browse" | "form" | "success">("browse");
    const [ticketId, setTicketId] = useState("");
    const [copied, setCopied] = useState(false);
    const [searchQuery, setSearchQuery] = useState("");
    const [openFaq, setOpenFaq] = useState<number | null>(null);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);
    // const [formData, setFormData] = useState({ subject: "", category: "technical", message: "" });
    const [formData, setFormData] = useState({
        name: "",
        email: "",
        phone: "",
        subject: "",
        category: "technical",
        message: ""
    });
    // const filteredFaqs = useMemo(() =>
    //     faqs.filter(f => f.question.toLowerCase().includes(searchQuery.toLowerCase())),
    //     [searchQuery]);

    const filteredFaqs = useMemo(() => {
        const query = searchQuery.toLowerCase().trim();

        if (!query) return faqs;

        return faqs.filter(f => {
            const matchesQuestion = f.question.toLowerCase().includes(query);
            const matchesKeywords = f.keywords.some(keyword => keyword.toLowerCase().includes(query));

            return matchesQuestion || matchesKeywords;
        });
    }, [searchQuery]);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsSubmitting(true);
        const device_id = localStorage.getItem("device_id")
        try {
            const response = await fetch(`${API_URL}/api/support/ticket/create/`, {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-Device-Id": device_id || "",
                },
                credentials: "include",
                body: JSON.stringify({
                    name: formData.name,
                    email: formData.email,
                    phone: formData.phone,
                    subject: formData.subject,
                    category: formData.category,
                    description: formData.message,
                }),
            });

            const data = await response.json();
            console.log(data.error)
            if (response.ok) {
                // Use the ticket_number from your API response
                setTicketId(data.ticket_number);
                setToast({
                    type: 'success',
                    message: data.message || "Ticket created successfully!"
                });
                setStep("success");

                // Reset form
                setFormData({
                    name: "", email: "", phone: "",
                    subject: "", category: "technical", message: ""
                });
            } else {
                setToast({
                    type: 'error',
                    message: data.error || "Failed to submit ticket"
                })
            }

        } catch (error: any) {
            setToast({
                type: 'error',
                message: "Something went wrong. Please try again."
            });
        } finally {
            setIsSubmitting(false);
        }
    };

    const copyToClipboard = () => {
        navigator.clipboard.writeText(ticketId);
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
    };

    return (
        <div className="min-h-screen bg-background mt-[60px]">
            <BasicNavbar />
            {/* Hero Section */}
            <div className="bg-background py-20 px-6 border-b border-white/5 relative overflow-hidden">
                <div className="max-w-4xl mx-auto text-center relative z-10">
                    <div className="flex justify-center mb-6">
                        <div className="w-12 h-12 bg-primary/20 rounded-full flex items-center justify-center border border-primary/30">
                            <HelpCircle className="text-primary" size={28} />
                        </div>
                    </div>
                    <h1 className="text-2xl md:text-3xl font-semibold tracking-tight mb-8">How can we help you?</h1>

                    <div className="relative max-w-2xl mx-auto">
                        <Search className="absolute left-5 top-1/2 -translate-y-1/2 text-muted-foreground" size={20} />
                        <input
                            className="w-full pl-14 pr-6 py-4 bg-secondary/90 border-none rounded-2xl outline-none focus:ring-2 focus:ring-primary/40 transition-all text-lg text-white placeholder:text-gray-500"
                            placeholder="Describe your issue"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                        />
                    </div>
                </div>
                <div className="absolute bottom-0 left-0 w-full h-24 opacity-20 pointer-events-none bg-gradient-to-t from-primary/20 to-transparent" />
            </div>

            <div className="max-w-4xl mx-auto md:p-10 mt-8 relative z-20">
                {step === "browse" ? (
                    <div className="p-6 md:p-0 space-y-8 animate-in fade-in slide-in-from-bottom-4 duration-500">
                        {/* FAQ List */}
                        <div className="bg-background border border-border rounded-3xl shadow-xl overflow-hidden">
                            <div className="p-6 border-b border-border bg-secondary/5">
                                <h2 className="font-bold text-lg">Popular help topics</h2>
                            </div>
                            <div className="divide-y divide-border">
                                {filteredFaqs.map((faq) => (
                                    <div key={faq.id} className="group">
                                        <button
                                            onClick={() => setOpenFaq(openFaq === faq.id ? null : faq.id)}
                                            className="w-full flex items-center justify-between p-6 text-left hover:bg-secondary/10 transition-colors"
                                        >
                                            <span className="font-medium">{faq.question}</span>
                                            <ChevronDown size={20} className={cn("transition-transform text-muted-foreground", openFaq === faq.id && "rotate-180")} />
                                        </button>
                                        {openFaq === faq.id && (
                                            <div className="px-6 pb-6 animate-in slide-in-from-top-2">
                                                <p className="text-muted-foreground text-sm leading-relaxed mb-4">{faq.answer}</p>
                                                <button className="text-xs font-semibold text-primary hover:underline flex items-center gap-1">
                                                    Was this helpful? <CheckCircle2 size={14} />
                                                </button>
                                            </div>
                                        )}
                                    </div>
                                ))}
                            </div>
                        </div>

                        {/* Step-up CTA */}
                        <div className="bg-primary/5 border border-primary/10 rounded-3xl p-8 flex flex-col md:flex-row items-center justify-between gap-6">
                            <div className="flex items-center gap-4">
                                <div className="p-3 bg-primary/10 rounded-2xl text-primary">
                                    <MessageSquare size={24} />
                                </div>
                                <div>
                                    <h3 className="font-bold">Still need help?</h3>
                                    <p className="text-sm text-muted-foreground">If you couldn't find an answer, please contact us.</p>
                                </div>
                            </div>
                            <button
                                onClick={() => setStep("form")}
                                className="bg-primary text-white px-8 py-3 rounded-xl font-bold text-sm flex items-center gap-2 hover:opacity-90 transition-all active:scale-95 w-full md:w-auto"
                            >
                                Contact Support <ArrowRight size={16} />
                            </button>
                        </div>

                        {/* Support Cards Grid */}
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-6">
                            <a
                                href="mailto:support@yourdomain.com"
                                className="flex items-center gap-4 p-5 bg-secondary/10 border border-border rounded-2xl hover:bg-secondary/20 transition-all group"
                            >
                                <div className="p-3 bg-background rounded-xl text-primary group-hover:scale-110 transition-transform">
                                    <Send size={20} />
                                </div>
                                <div>
                                    <p className="text-xs font-bold uppercase tracking-wider text-muted-foreground">Email Us</p>
                                    <p className="text-sm font-semibold">support@yourdomain.com</p>
                                </div>
                            </a>

                            <a
                                href="tel:+15550000000"
                                className="flex items-center gap-4 p-5 bg-secondary/10 border border-border rounded-2xl hover:bg-secondary/20 transition-all group"
                            >
                                <div className="p-3 bg-background rounded-xl text-primary group-hover:scale-110 transition-transform">
                                    <LifeBuoy size={20} />
                                </div>
                                <div>
                                    <p className="text-xs font-bold uppercase tracking-wider text-muted-foreground">Call Us (24/7)</p>
                                    <p className="text-sm font-semibold">+1 (555) 000-0000</p>
                                </div>
                            </a>
                        </div>
                    </div>
                ) : step === "form" ? (
                    /* Submission Form - Step 2 */
                    <div className="animate-in fade-in zoom-in-95 duration-300">
                        <button
                            onClick={() => setStep("browse")}
                            className="flex items-center gap-2 text-sm text-muted-foreground hover:text-foreground mb-6 transition-colors px-6 md:px-0"
                        >
                            <X size={16} /> Cancel and go back
  </button>

                        <div className="bg-background md:border border-border md:rounded-3xl p-6 md:p-8 md:shadow-2xl">
                            <h2 className="text-2xl font-bold mb-2">Submit a request</h2>
                            <p className="text-muted-foreground text-sm mb-8">Please provide your details so we can assist you better.</p>

                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    {/* Name Field */}
                                    <div className="space-y-2">
                                        <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Full Name</label>
                                        <input
                                            required
                                            className="mt-2 w-full p-4 bg-secondary/20 border border-border rounded-2xl outline-none focus:ring-2 focus:ring-primary/20 transition-all"
                                            placeholder="John Doe"
                                            value={formData.name}
                                            onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                        />
                                    </div>

                                    {/* Email Field */}
                                    <div className="space-y-2">
                                        <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Email Address</label>
                                        <input
                                            required
                                            type="email"
                                            className="mt-2 w-full p-4 bg-secondary/20 border border-border rounded-2xl outline-none focus:ring-2 focus:ring-primary/20 transition-all"
                                            placeholder="john@example.com"
                                            value={formData.email}
                                            onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                                        />
                                    </div>

                                    {/* Phone Number Field */}
                                    <div className="space-y-2">
                                        <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Phone Number</label>
                                        <input
                                            type="tel"
                                            className="mt-2 w-full p-4 bg-secondary/20 border border-border rounded-2xl outline-none focus:ring-2 focus:ring-primary/20 transition-all"
                                            placeholder="+91 00000-00000"
                                            value={formData.phone}
                                            onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                                        />
                                    </div>

                                    {/* Category Field */}
                                    <div className="space-y-2">
                                        <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Category</label>
                                        <div className="mt-2 flex gap-2 p-1 bg-secondary/30 rounded-2xl border border-border">
                                            {['Technical', 'Account'].map((cat) => (
                                                <button
                                                    key={cat}
                                                    type="button"
                                                    onClick={() => setFormData({ ...formData, category: cat.toLowerCase() })}
                                                    className={cn(
                                                        "flex-1 py-4 text-xs font-bold rounded-xl transition-all",
                                                        formData.category === cat.toLowerCase()
                                                            ? "bg-background text-primary shadow-sm"
                                                            : "text-muted-foreground hover:text-foreground"
                                                    )}
                                                >
                                                    {cat}
                                                </button>
                                            ))}
                                        </div>
                                    </div>
                                </div>

                                {/* Subject Field (Full Width) */}
                                <div className="space-y-2">
                                    <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Subject</label>
                                    <input
                                        required
                                        className="mt-2 w-full p-4 bg-secondary/20 border border-border rounded-2xl outline-none focus:ring-2 focus:ring-primary/20 transition-all"
                                        placeholder="e.g., Access Issue"
                                        value={formData.subject}
                                        onChange={(e) => setFormData({ ...formData, subject: e.target.value })}
                                    />
                                </div>

                                {/* Message Field (Full Width) */}
                                <div className="space-y-2">
                                    <label className="text-xs font-bold uppercase text-muted-foreground ml-1">Description</label>
                                    <textarea
                                        required
                                        rows={5}
                                        className="mt-2 w-full p-4 bg-secondary/20 border border-border rounded-2xl outline-none focus:ring-2 focus:ring-primary/20 transition-all resize-none"
                                        placeholder="Tell us about the issue..."
                                        value={formData.message}
                                        onChange={(e) => setFormData({ ...formData, message: e.target.value })}
                                    />
                                </div>

                                <div className="flex justify-end">
                                    <button
                                        disabled={isSubmitting}
                                        className="w-full md:w-auto px-10 py-4 bg-primary text-white rounded-2xl font-bold flex items-center justify-center gap-2 hover:opacity-90 transition-all disabled:opacity-50"
                                    >
                                        {isSubmitting ? <Loader2 size={18} className="animate-spin" /> : <Send size={18} />}
          Submit Ticket
        </button>
                                </div>
                            </form>
                        </div>
                    </div>
                ) : (
                            /* Success State - Reference Number */
                            <div className="p-6 md:p-0 animate-in zoom-in-95 duration-500">
                                <div className="bg-background border border-border rounded-3xl p-8 md:p-12 shadow-2xl text-center flex flex-col items-center">
                                    <div className="w-20 h-20 bg-emerald-500/10 rounded-full flex items-center justify-center text-emerald-500 mb-6">
                                        <CheckCircle2 size={40} />
                                    </div>
                                    <h2 className="text-2xl font-bold mb-3">Request Received</h2>
                                    <p className="text-muted-foreground max-w-sm mb-8">
                                        We have received your query. Our team will review it and get back to you shortly.
                </p>

                                    <div className="w-full max-w-xs space-y-2 mb-8">
                                        <p className="text-xs font-bold uppercase tracking-widest text-muted-foreground">Reference Number</p>
                                        <div className="flex items-center gap-2 p-4 bg-secondary/30 border border-border rounded-2xl justify-center group relative">
                                            <span className="text-xl font-mono font-bold tracking-wider">{ticketId}</span>
                                            <button
                                                onClick={copyToClipboard}
                                                className="p-2 hover:bg-primary/10 rounded-lg text-primary transition-colors"
                                                title="Copy Reference Number"
                                            >
                                                {copied ? <Check size={18} className="text-emerald-500" /> : <Copy size={18} />}
                                            </button>
                                        </div>
                                    </div>
                                    {/* Quick Contact Footer */}
                                    <div className="w-full pt-8 border-t border-border flex flex-col md:flex-row justify-center gap-8 mb-8">
                                        <div className="flex flex-col items-center gap-1">
                                            <p className="text-[10px] font-bold uppercase text-muted-foreground">Urgent Follow-up</p>
                                            <a href="tel:+15550000000" className="text-sm font-bold hover:text-primary">+1 (555) 000-0000</a>
                                        </div>
                                        <div className="flex flex-col items-center gap-1">
                                            <p className="text-[10px] font-bold uppercase text-muted-foreground">General Info</p>
                                            <a href="mailto:info@domain.com" className="text-sm font-bold hover:text-primary">info@domain.com</a>
                                        </div>
                                    </div>
                                    <button
                                        onClick={() => setStep("browse")}
                                        className="flex items-center gap-2 text-sm font-semibold text-primary hover:underline"
                                    >
                                        <ArrowLeft size={16} /> Back to Help Center
                </button>
                                </div>
                            </div>
                        )}
            </div>

            {toast && <Toast message={toast.message} type={toast.type} onClose={() => setToast(null)} />}
        </div>
    );
}
