import { NextResponse } from "next/server";
import bcrypt from "bcryptjs";

import { connectDB } from "@/lib/db";
import User from "@/models/User";
import { registerSchema } from "@/validators/register.schema";

export async function POST(req: Request) {
  try {
    const body = await req.json();

    // Validate input
    const parsed = registerSchema.safeParse(body);
    if (!parsed.success) {
      return NextResponse.json(
        { error: "Invalid input", details: parsed.error.flatten() },
        { status: 400 }
      );
    }

    const {
      firstName,
      lastName,
      phoneNumber,
      primaryEmail,
      recoveryPhone,
      recoveryEmail,
      address,
      password,
    } = parsed.data;

    await connectDB();

    // Check duplicates
    const existingUser = await User.findOne({
      $or: [
        { primaryEmail: primaryEmail.toLowerCase() },
        { phoneNumber },
      ],
    });

    if (existingUser) {
      return NextResponse.json(
        { error: "User already exists" },
        { status: 409 }
      );
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(password, 12);

    await User.create({
      firstName,
      lastName,
      phoneNumber,
      primaryEmail: primaryEmail.toLowerCase(),
      recoveryPhone,
      recoveryEmail,
      address,
      password: hashedPassword,
    });

    return NextResponse.json(
      { message: "User registered successfully" },
      { status: 201 }
    );
  } catch (error) {
    console.error("REGISTER ERROR:", error);
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}
