"use client";
import React, { useState, useRef, useEffect } from "react";
import { Country, COUNTRIES } from "@/lib/country-data";

interface CountrySelectorProps {
  onChange?: (country: Country) => void;
  defaultValue?: string;
}

export default function CountrySelector({ onChange, defaultValue }: CountrySelectorProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [selected, setSelected] = useState<Country>(
    COUNTRIES.find((c) => c.code === defaultValue) || COUNTRIES.find((c) => c.dial === defaultValue) || COUNTRIES.find((c) => c.dial === "91") ||COUNTRIES[0]

  );
  const [search, setSearch] = useState("");
  const dropdownRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(e.target as Node)) setIsOpen(false);
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  const filtered = COUNTRIES.filter(
    (c) => c.name.toLowerCase().includes(search.toLowerCase()) || c.dial.includes(search)
  );

  return (
    <div className="relative w-full max-w-[160px]" ref={dropdownRef}>
      {/* Smaller Trigger: Reduced padding (py-2, px-3) and font size */}
      <button
        type="button"
        onClick={() => setIsOpen(!isOpen)}
        style={{ backgroundColor:'var(--background)', color: 'var(--foreground)', borderColor: 'var(--border)' }}
        className={`flex items-center justify-between w-full px-3 py-2 rounded-lg transition-all text-sm ${
          isOpen ? " " : "hover:border-zinc-500"
        }`}
      >
        <div className="flex items-center gap-2" style={{alignItems:"center"}}>
        <img src={selected.flag_icon} width="20" alt="IN" className="mt-[4px]"/>
          {/* <span className="text-lg">{selected.flag}</span> */}
          <span className="font-bold">+{selected.dial}</span>
        </div>
 
        <svg
          className={`w-3 h-3 transition-transform ${isOpen ? "rotate-180" : ""}`}
          style={{ color: 'var(--muted-foreground)' }}
          fill="none" viewBox="0 0 24 24" stroke="currentColor"
        >
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={3} d="M19 9l-7 7-7-7" />
        </svg>

      </button>

      {/* Condensed Dropdown */}
      {isOpen && (
        <div 
          style={{ backgroundColor: 'var(--popover)', borderColor: 'var(--border)', color: 'var(--popover-foreground)' }}
          className="absolute z-50 w-64 mt-1 border rounded-xl shadow-xl overflow-hidden animate-in fade-in slide-in-from-top-1"
        >
          <div className="p-2" style={{ borderBottom: '1px solid var(--border)', backgroundColor: 'var(--muted)' }}>
            <input
              type="text"
              placeholder="Search..."
              style={{ backgroundColor: 'var(--background)', color: 'var(--foreground)', borderColor: 'var(--input)' }}
              className="w-full px-3 py-1.5 text-xs border rounded-md focus:outline-none"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              autoFocus
            />
          </div>

          <ul className="max-h-48 bg-background overflow-y-auto custom-scrollbar py-1">
            {filtered.length > 0 ? (
              filtered.map((c) => (
                <li
                  key={c.code}
                  onClick={() => {
                    setSelected(c);
                    setIsOpen(false);
                    setSearch("");
                    onChange?.(c);
                  }}
                  className="flex items-center justify-between px-3 py-1.5 cursor-pointer hover:bg-[var(--accent)] transition-colors"
                >
                  <div className="flex items-center gap-2">
                    <img src={c.flag_icon} width="20" alt="IN" className="mt-[4px]"/>
                    <span className="text-base">{c.flag}</span>
                    <span className="text-xs font-medium truncate max-w-[120px]">{c.name}</span>
                  </div>
                  <span className="text-[10px] opacity-70 font-mono">+{c.dial}</span>
                </li>
              ))
            ) : (
              <li className="px-3 py-4 text-[10px] text-center opacity-50">No results</li>
            )}
          </ul>
        </div>
      )}
    </div>
  );
}