import React, { useState } from 'react';
import { ChevronLeft, ChevronRight, ChevronDown } from 'lucide-react';

const CustomCalendar = ({ selectedDate, onSelect,  }) => {
  // Convert "2002-08-10" string to Date object for logic
  // We use UTC components or split to avoid timezone shift issues
  const parseISO = (str) => {
    if (!str) return new Date();
    const [y, m, d] = str.split('-').map(Number);
    return new Date(y, m - 1, d);
  };

  const [viewDate, setViewDate] = useState(parseISO(selectedDate));
  const [viewMode, setViewMode] = useState('days'); // 'days' | 'months' | 'years'

  const month = viewDate.getMonth();
  const year = viewDate.getFullYear();

  // Navigation Logic
  const handleSelect = (d, m, y) => {
    // Format back to "YYYY-MM-DD"
    const formatted = `${y}-${String(m + 1).padStart(2, '0')}-${String(d).padStart(2, '0')}`;
    onSelect(formatted);
  };

  const changeMonth = (offset) => setViewDate(new Date(year, month + offset, 1));
  
  const months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];

  // Grid Calculation
  const daysInMonth = new Date(year, month + 1, 0).getDate();
  const firstDayIndex = new Date(year, month, 1).getDay();
  const shift = firstDayIndex === 0 ? 6 : firstDayIndex - 1;

  return (
    <div className="w-80 p-6 bg-background rounded-[32px] border border-border shadow-2xl select-none">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <button onClick={() => changeMonth(-1)} className="p-2 hover:bg-secondary rounded-full text-muted-foreground transition-colors">
          <ChevronLeft size={20} />
        </button>
        
        <div className="flex gap-1">
          <button 
            onClick={() => setViewMode(viewMode === 'months' ? 'days' : 'months')}
            className="text-lg font-bold text-foreground hover:text-primary transition-colors"
          >
            {months[month]}
          </button>
          <button 
            onClick={() => setViewMode(viewMode === 'years' ? 'days' : 'years')}
            className="text-lg font-bold text-foreground hover:text-primary transition-colors"
          >
            {year}
          </button>
        </div>

        <button onClick={() => changeMonth(1)} className="p-2 hover:bg-secondary rounded-full text-muted-foreground transition-colors">
          <ChevronRight size={20} />
        </button>
      </div>

      {/* DAYS VIEW */}
      {viewMode === 'days' && (
        <>
          <div className="grid grid-cols-7 gap-1 mb-4 text-center">
            {['M', 'T', 'W', 'T', 'F', 'S', 'S'].map((d, i) => (
              <span key={i} className="text-[10px] font-bold text-muted-foreground/50 uppercase">{d}</span>
            ))}
          </div>
          <div className="grid grid-cols-7 gap-1">
            {Array.from({ length: shift }).map((_, i) => <div key={`e-${i}`} />)}
            {Array.from({ length: daysInMonth }).map((_, i) => {
                const day = i + 1;
                const isSelected = selectedDate === `${year}-${String(month+1).padStart(2,'0')}-${String(day).padStart(2,'0')}`;
                return (
                  <button
                    key={day}
                    onClick={() => handleSelect(day, month, year)}
                    className={`h-9 w-9 flex items-center justify-center text-sm rounded-xl transition-all
                      ${isSelected ? 'bg-foreground text-background font-bold scale-105 shadow-md' : 'hover:bg-secondary text-foreground'}`}
                  >
                    {day}
                  </button>
                );
            })}
          </div>
        </>
      )}

      {/* MONTHS VIEW */}
      {viewMode === 'months' && (
        <div className="grid grid-cols-3 gap-3">
          {months.map((m, i) => (
            <button
              key={m}
              onClick={() => { setViewDate(new Date(year, i, 1)); setViewMode('days'); }}
              className={`py-3 text-sm rounded-xl transition-colors ${i === month ? 'bg-primary text-primary-foreground' : 'hover:bg-secondary'}`}
            >
              {m}
            </button>
          ))}
        </div>
      )}

      {/* YEARS VIEW */}
      {viewMode === 'years' && (
        <div className="grid grid-cols-3 gap-2 h-64 overflow-y-auto no-scrollbar pr-1">
          {Array.from({ length: 100 }).map((_, i) => {
            const y = new Date().getFullYear() - i;
            return (
              <button
                key={y}
                onClick={() => { setViewDate(new Date(y, month, 1)); setViewMode('days'); }}
                className={`py-2 text-sm rounded-lg transition-colors ${y === year ? 'bg-primary text-primary-foreground' : 'hover:bg-secondary'}`}
              >
                {y}
              </button>
            );
          })}
        </div>
      )}
    </div>
  );
};

export default CustomCalendar