"use client";

import React, { useState, useEffect, useRef } from "react";
import {
  X, Check, Copy, Upload, Link,
  ChevronDown, ShieldAlert, Eye, EyeOff,
  Image as ImageIcon, Globe, Trash2, Power
} from "lucide-react";
import { cn } from "@/lib/utils";
import { adminAuthService } from '@/lib/adminAuthService';

interface AppData {
  id?: number | string;
  appId?: string;
  name: string;
  category: string;
  status: "Connected" | "Disabled";
  icon: string;
  iconType: "link" | "upload";
  description: string;
  clientId?: string;
  clientSecret?: string;
  callbackUrl?: string;
  redirectUri?: string;
}

// const APP_TEMPLATES = [
//   { id: "strp_01", name: "Stripe Connect" },
//   { id: "auth_01", name: "Auth0 Identity" },
//   { id: "goog_01", name: "Google Cloud" },
//   { id: "cust_01", name: "Custom Integration" },
// ];


const DUMMY_APP_DATA = {
  appId: "",
  name: "",
  category: "General",
  status: "Connected",
  icon: "",
  iconType: "link",
  description: "",
  clientId: "client_" + Math.random().toString(36).substring(7),
  clientSecret: "sk_live_" + Math.random().toString(36).substring(2),
  callbackUrl: "",
  redirectUri: "https://api.yoursite.com/v1/callback",
}


const API_BASE_URL = `${process.env.NEXT_PUBLIC_API_BASE_URL}/api/admin`;

export default function InstallAppModal({ isOpen, onClose, initialData, onDelete }: any) {

  // 1. All State Hooks (Always at the top)
  const [isAppIdOpen, setIsAppIdOpen] = useState(false);
  const [iconMode, setIconMode] = useState<"link" | "upload">("link");
  const [showSecret, setShowSecret] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [previewUrl, setPreviewUrl] = useState<string>("");
  const [formData, setFormData] = useState<AppData>(DUMMY_APP_DATA);
  const [appIDs, setAppIDs] = useState([])
  const [copiedKey, setCopiedKey] = useState<string | null>(null);
  const [message, setMessage] = useState<{
    text: string;
    type: "error" | "warning" | "success"
  } | null>(null);
  const [isDeleting, setIsDeleting] = useState(false);
  const [isDeleteConfirmOpen, setIsDeleteConfirmOpen] = useState(false);
  const [deleteStatus, setDeleteStatus] = useState<"idle" | "loading" | "success">("idle");
  const [isRotateConfirmOpen, setIsRotateConfirmOpen] = useState(false);
  const [rotateStatus, setRotateStatus] = useState<"idle" | "loading" | "success">("idle");


  // const getAppIDs = async () => {
  //   try {
  //     const response = await fetch(`${API_BASE_URL}/apps/available/ids`, {
  //       method: "GET",
  //       credentials: "include"
  //     });

  //     const data = await response.json();

  //     if (response.ok && Array.isArray(data) && data.length > 0) {
  //       setAppIDs(data);

  //       // Only set default appId if we are creating a NEW app (no initialData)
  //       if (!initialData) {
  //         setFormData(prev => ({
  //           ...prev,
  //           appId: data[0].id // Safely access the first ID
  //         }));
  //       }
  //     }
  //   } catch (error) {
  //     setMessage({ text: "Something Went Wrong", type: "error" })
  //   }
  // };



  const getAppIDs = async () => {
    try {
      // 1. Call the service
      const result = await adminAuthService.getAvailableAppIDs();

      // 2. Process data only on success
      if (result.success && Array.isArray(result.data) && result.data.length > 0) {
        setAppIDs(result.data);

        // Logic for setting default selection for NEW apps
        if (!initialData) {
          setFormData(prev => ({
            ...prev,
            appId: result.data![0].id
          }));
        }
      } else if (!result.success) {
        // Handle API-specific errors (e.g., 403 Forbidden)
        setMessage({ text: result.message || "Failed to load IDs", type: "error" });
      }
    } catch (error) {
      // Handle network/unhandled errors
      setMessage({ text: "Something Went Wrong", type: "error" });
    }
  };
  // 2. All Ref Hooks
  const fileInputRef = useRef<HTMLInputElement>(null);

  // 3. All Effect Hooks

  useEffect(() => {
    getAppIDs()
    if (initialData) {
      setFormData({
        appId: initialData.system_app_id,
        name: initialData.display_name,
        category: "General",
        status: initialData.enabled ? "Connected" : "Disabled",
        icon: initialData.icon.value,
        iconType: initialData.icon.type,
        description: initialData.description,
        clientId: initialData.client_id,
        clientSecret: initialData.client_secret,
        callbackUrl: initialData.callback_url,
        redirectUri: initialData.redirect_uri
      });
    } else {
      // setFormData(DUMMY_APP_DATA);
    }

  }, [initialData, isOpen]);

  useEffect(() => {
    return () => {
      if (previewUrl) URL.revokeObjectURL(previewUrl);
    };
  }, [previewUrl]);

  // 4. Early return MUST come AFTER all hooks are declared
  if (!isOpen) return null;

  // 5. Logic Handlers
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setSelectedFile(file);
      setPreviewUrl(URL.createObjectURL(file));
      setFormData({ ...formData, iconType: "upload" });
    }
  };

  // const handleAppSubmit = async () => {
  //   if (!formData.name || !formData.appId) {
  //     // alert("Please fill in the required fields.");
  //     setMessage({ text: "Please fill in all required fields.", type: "warning" });
  //     return;
  //   }

  //   setIsSubmitting(true);
  //   try {
  //     const data = new FormData();
  //     data.append("system_app_id", String(formData.appId));
  //     data.append("display_name", formData.name);
  //     data.append("description", formData.description);
  //     data.append("callback_url", formData.callbackUrl || "");
  //     data.append("icon_type", iconMode === "link" ? "url" : "upload");
  //     data.append("enabled", String(formData.status === "Connected"));

  //     if (iconMode === "upload" && selectedFile) {
  //       data.append("icon", selectedFile);
  //     } else {
  //       data.append("icon_url", formData.icon);
  //     }

  //     console.log(data)
  //     const response = await fetch(`${API_BASE_URL}/apps/install/`, {
  //       method: "POST",
  //       credentials: "include",
  //       body: data,
  //     });

  //     const resData = await response.json()
  //     console.log(resData)
  //     if (!response.ok) { throw new Error("Installation failed") }
  //     else {
  //       setMessage({ text: resData.error || "Installation failed.", type: "error" });
  //     }
  //     onClose();
  //   } catch (error) {
  //     setMessage({ text: "Internal server error Installation failed.", type: "error" });
  //   } finally {
  //     setIsSubmitting(false);
  //   }
  // };



  const handleAppSubmit = async () => {
    // 1. Local Validation
    if (!formData.name || !formData.appId) {
      setMessage({ text: "Please fill in all required fields.", type: "warning" });
      return;
    }

    setIsSubmitting(true);
    try {
      // 2. Build FormData
      const data = new FormData();
      data.append("system_app_id", String(formData.appId));
      data.append("display_name", formData.name);
      data.append("description", formData.description);
      data.append("callback_url", formData.callbackUrl || "");
      data.append("icon_type", iconMode === "link" ? "url" : "upload");
      data.append("enabled", String(formData.status === "Connected"));

      if (iconMode === "upload" && selectedFile) {
        data.append("icon", selectedFile);
      } else {
        data.append("icon_url", formData.icon);
      }

      // 3. Call the Service
      const result = await adminAuthService.installApp(data);

      // 4. Handle Standardized Response
      if (result.success) {
        setMessage({ text: "App installed successfully!", type: "success" });
        // onClose();
        handleClose()
      } else {
        setMessage({
          text: result.message || "Installation failed.",
          type: "error"
        });
      }
    } catch (error) {
      setMessage({ text: "Internal server error. Installation failed.", type: "error" });
    } finally {
      setIsSubmitting(false);
    }
  };

  const copyToClipboard = (text: string, key: string) => {
    navigator.clipboard.writeText(text);
    setCopiedKey(key);
    // Revert the icon back after 2 seconds
    setTimeout(() => {
      setCopiedKey(null);
    }, 2000);
  };

  const toggleStatus = () => {
    setFormData(prev => ({
      ...prev,
      status: prev.status === "Connected" ? "Disabled" : "Connected"
    }));
  };


  // --- Reset Function ---
  const resetForm = () => {
    setFormData(DUMMY_APP_DATA);
    setSelectedFile(null);
    if (previewUrl) {
      URL.revokeObjectURL(previewUrl);
      setPreviewUrl("");
    }
    setIconMode("link");
    setIsAppIdOpen(false);
  };

  // Handle closing wrapper
  const handleClose = () => {
    resetForm();
    setMessage(null)
    onClose();
  };



  // const handleAppUpdate = async () => {
  //   if (!formData.name) {
  //     // alert("Display name is required.");
  //     setMessage({ text: "Display name is required.", type: "warning" });
  //     return;
  //   }

  //   setIsSubmitting(true);

  //   try {
  //     const data = new FormData();

  //     // 1. Map fields to the API expectations
  //     data.append("display_name", formData.name);
  //     data.append("description", formData.description);
  //     data.append("callback_url", formData.callbackUrl || "");
  //     data.append("enabled", String(formData.status === "Connected"));

  //     console.log(String(formData.status === "Connected"))
  //     // 2. Handle Icon Logic
  //     data.append("icon_type", iconMode === "link" ? "url" : "upload");

  //     if (iconMode === "upload" && selectedFile) {
  //       // If a new file was picked, upload the real file
  //       data.append("icon", selectedFile);
  //     } else {
  //       // Otherwise, send the URL (either existing or newly entered)
  //       data.append("icon_url", formData.icon);
  //     }

  //     // 3. Execute PUT request
  //     const response = await fetch(`${API_BASE_URL}/apps/update/${formData.appId}/`, {
  //       method: "PATCH", // or "PATCH" depending on your backend preference
  //       credentials: "include",
  //       body: data,
  //     });

  //     if (!response.ok) {
  //       const errorData = await response.json();
  //       console.log(errorData)
  //       throw new Error(errorData.error || "Failed to update application.");
  //     }

  //     const result = await response.json();
  //     console.log("Update Success:", result);

  //     handleClose(); // Reset state and close modal
  //   } catch (error: any) {
  //     setMessage({ text: error.message || "An error occurred during update.", type: "error" });
  //   } finally {
  //     setIsSubmitting(false);
  //   }
  // };


  const handleAppUpdate = async () => {
    if (!formData.name) {
      setMessage({ text: "Display name is required.", type: "warning" });
      return;
    }

    setIsSubmitting(true);

    try {
      const data = new FormData();
      data.append("display_name", formData.name);
      data.append("description", formData.description);
      data.append("callback_url", formData.callbackUrl || "");
      data.append("enabled", String(formData.status === "Connected"));
      data.append("icon_type", iconMode === "link" ? "url" : "upload");

      if (iconMode === "upload" && selectedFile) {
        data.append("icon", selectedFile);
      } else {
        data.append("icon_url", formData.icon);
      }

      // 1. Call the service
      const result = await adminAuthService.updateApp(formData.appId, data);

      // 2. Handle outcome
      if (result.success) {
        // console.log("Update Success:", result.data);
        handleClose(); // Success!
      } else {
        // result.message automatically extracts errorData.error from your old logic
        setMessage({
          text: result.message || "Failed to update application.",
          type: "error"
        });
      }
    } catch (error: any) {
      setMessage({
        text: error.message || "An error occurred during update.",
        type: "error"
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  // Update the Submit handler to choose between Install or Update
  const onFormSubmit = () => {
    if (initialData) {
      handleAppUpdate();
    } else {
      handleAppSubmit(); // Your existing POST logic
    }
  };

  // const handleConfirmDelete = async () => {
  //   setDeleteStatus("loading");
  //   try {
  //     const response = await fetch(`${API_BASE_URL}/apps/${formData.appId}/uninstall/`, {
  //       method: "DELETE",
  //       credentials: "include",
  //     });

  //     if (!response.ok) throw new Error("Uninstall failed");

  //     // Show "Deleted" text
  //     setDeleteStatus("success");

  //     // Notify parent to remove from list
  //     if (onDelete) onDelete(formData.appId);

  //     // Wait 1.5s, then close everything
  //     setTimeout(() => {
  //       setIsDeleteConfirmOpen(false);
  //       handleClose(); // Closes the main modal too
  //       setDeleteStatus("idle");
  //     }, 1500);

  //   } catch (error) {
  //     setDeleteStatus("idle");
  //     setMessage({ text: "Failed to delete the app.", type: "error" });
  //     setIsDeleteConfirmOpen(false);
  //   }
  // };


  const handleConfirmDelete = async () => {
    setDeleteStatus("loading");

    try {
      // 1. Call the service
      const result = await adminAuthService.uninstallApp(formData.appId);

      if (result.success) {
        // 2. Show "Deleted" UI state
        setDeleteStatus("success");

        // 3. Notify parent to update the list locally
        if (onDelete) onDelete(formData.appId);

        // 4. Delayed UI Reset
        setTimeout(() => {
          setIsDeleteConfirmOpen(false);
          handleClose();
          setDeleteStatus("idle");
        }, 1500);
      } else {
        // Handle specific backend rejection (e.g., app is protected)
        setDeleteStatus("idle");
        setMessage({
          text: result.message || "Failed to delete the app.",
          type: "error"
        });
        setIsDeleteConfirmOpen(false);
      }
    } catch (error) {
      setDeleteStatus("idle");
      setMessage({ text: "Network error occurred.", type: "error" });
      setIsDeleteConfirmOpen(false);
    }
  };

  // const handleRotateSecret = async () => {
  //   setRotateStatus("loading");
  //   try {
  //     const response = await fetch(`${API_BASE_URL}/apps/${formData.appId}/rotate-secret/`, {
  //       method: "POST", // Usually POST for action-based routes
  //       credentials: "include",
  //     });

  //     if (!response.ok) throw new Error("Failed to rotate secret");

  //     const data = await response.json();

  //     // 1. Update the local state with the NEW secret from the backend
  //     setFormData(prev => ({
  //       ...prev,
  //       clientSecret: data.client_secret // Ensure this matches your API response key
  //     }));

  //     setRotateStatus("success");

  //     // 2. Close the confirmation modal after a short delay
  //     setTimeout(() => {
  //       setIsRotateConfirmOpen(false);
  //       setRotateStatus("idle");
  //       setMessage({ text: "Client secret rotated successfully!", type: "success" });
  //     }, 1500);

  //   } catch (error: any) {
  //     setRotateStatus("idle");
  //     setIsRotateConfirmOpen(false);
  //     setMessage({ text: error.message || "Rotation failed", type: "error" });
  //   }
  // };



  const handleRotateSecret = async () => {
    setRotateStatus("loading");
    try {
      // 1. Call the service
      const result = await adminAuthService.rotateAppSecret(formData.appId);

      if (result.success && result.data) {
        // 2. Update local state with the NEW secret
        setFormData(prev => ({
          ...prev,
          clientSecret: result.data.client_secret
        }));

        setRotateStatus("success");

        // 3. UI Cleanup
        setTimeout(() => {
          setIsRotateConfirmOpen(false);
          setRotateStatus("idle");
          setMessage({ text: "Client secret rotated successfully!", type: "success" });
        }, 1500);
      } else {
        throw new Error(result.message || "Rotation failed");
      }

    } catch (error: any) {
      setRotateStatus("idle");
      setIsRotateConfirmOpen(false);
      setMessage({ text: error.message || "Rotation failed", type: "error" });
    }
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-background/80 backdrop-blur-md md:p-6">
      {/* Hidden File Input */}
      <input
        type="file"
        ref={fileInputRef}
        className="hidden"
        accept="image/png, image/jpeg, image/svg+xml"
        onChange={handleFileChange}
      />

      <div className="bg-background w-full h-full md:h-auto md:max-h-[90vh] md:max-w-3xl md:rounded-[3rem] md:border border-border shadow-2xl flex flex-col overflow-hidden animate-in fade-in zoom-in duration-200">

        {/* ... Header Content ... */}
        <div>
          <div className="p-6 md:p-10 border-b border-border flex justify-between items-center">
            <div>
              <h2 className="text-3xl font-bold tracking-tight">{initialData ? "App Settings" : "Install App"}</h2>
              <p className="text-muted-foreground text-sm mt-1">Configure your workspace connection.</p>
            </div>
            <div className="flex items-center gap-2">
              {initialData && (
                <button
                  onClick={() => setIsDeleteConfirmOpen(true)} // Just open the confirm modal
                  className="p-3 text-destructive hover:bg-destructive/10 rounded-full transition-all"
                  title="Uninstall Integration"
                >
                  <Trash2 size={22} />
                </button>
              )}
              <button onClick={handleClose} className="p-3 hover:bg-secondary rounded-full transition-all">
                <X size={24} />
              </button>
            </div>
          </div>
          {message && (
            <div className={cn(
              "flex items-start gap-3 p-4 rounded-2xl animate-in fade-in slide-in-from-top-2 mb-6",
              message.type === "error" && "bg-destructive/10 text-red-600 border border-destructive/20",
              message.type === "warning" && "bg-amber-500/10 text-amber-600 border border-amber-500/20",
              message.type === "success" && "bg-emerald-500/10 text-emerald-600 border border-emerald-500/20"
            )}>
              {message.type === "success" ? (
                <Check size={18} className="mt-0.5 shrink-0" />
              ) : (
                  <ShieldAlert size={18} className="mt-0.5 shrink-0" />
                )}

              <div className="flex-1 text-sm font-medium">
                {message.text}
              </div>

              <button
                onClick={() => setMessage(null)}
                className="opacity-70 hover:opacity-100 transition-opacity"
              >
                <X size={16} />
              </button>
            </div>
          )}
        </div>
        <div className="flex-1 overflow-y-auto p-6 md:p-10 space-y-8">

          {/* Status Toggle & App ID Row */}
          <div className="flex flex-col md:flex-row gap-6">
            <div className="flex-1 space-y-2">
              <label className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground ml-1">System App ID</label>
              <div className="relative">
                <button
                  onClick={() => setIsAppIdOpen(!isAppIdOpen)}
                  className="w-full flex items-center justify-between px-5 py-4 bg-secondary/40 border border-transparent hover:border-primary/20 rounded-2xl transition-all font-mono text-sm"
                >
                  {formData.appId}
                  <ChevronDown size={16} className={cn("transition-transform", isAppIdOpen && "rotate-180")} />
                </button>
                {isAppIdOpen && (
                  <div className="absolute top-full left-0 right-0 mt-2 bg-background border border-border rounded-2xl shadow-2xl z-20 py-2 animate-in slide-in-from-top-2">
                    {appIDs.map((t) => (
                      <button
                        key={t.id}
                        onClick={() => { setFormData({ ...formData, appId: t.id }); setIsAppIdOpen(false); }}
                        className="w-full px-5 py-3 text-left hover:bg-secondary text-sm font-mono flex justify-between items-center"
                      >
                        {t.id} <span className="text-[10px] opacity-50 uppercase tracking-tighter">{t.name}</span>
                      </button>
                    ))}
                  </div>
                )}
              </div>
            </div>

            {/* Status Switch */}
            <div className="space-y-2 min-w-[140px]">
              <label className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground ml-1">App Status</label>
              <button
                onClick={toggleStatus}
                className={cn(
                  "w-full flex items-center justify-center gap-2 px-5 py-4 rounded-2xl font-bold text-xs transition-all border",
                  formData.status === "Connected"
                    ? "bg-emerald-500/10 border-emerald-500/20 text-emerald-600"
                    : "bg-secondary/60 border-transparent text-muted-foreground"
                )}
              >
                <Power size={14} />
                {formData.status.toUpperCase()}
              </button>
            </div>
          </div>

          {/* App Name */}
          <div className="space-y-2">
            <label className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground ml-1">Display Name</label>
            <input
              type="text"
              value={formData.name}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              className="w-full px-5 py-4 bg-secondary/40 border-none rounded-2xl focus:ring-2 focus:ring-primary/20 outline-none"
              placeholder="e.g. Production Stripe"
            />
          </div>

          {/* App Description */}
          <div className="space-y-2">
            <label className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground ml-1">Description</label>
            <textarea
              value={formData.description}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              rows={3}
              className="w-full px-5 py-4 bg-secondary/40 border-none rounded-2xl focus:ring-2 focus:ring-primary/20 outline-none resize-none text-sm"
              placeholder="What does this integration do?"
            />
          </div>

          {/* App Icon Picker */}

          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <label className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground ml-1">App Icon</label>
              <div className="flex bg-secondary/50 p-1 rounded-xl">
                <button
                  onClick={() => setIconMode("link")}
                  className={cn("px-3 py-1.5 rounded-lg text-[10px] font-bold flex items-center gap-2 transition-all", iconMode === "link" ? "bg-background shadow-sm" : "opacity-50")}
                >
                  <Link size={12} /> URL
                </button>
                <button
                  onClick={() => setIconMode("upload")}
                  className={cn("px-3 py-1.5 rounded-lg text-[10px] font-bold flex items-center gap-2 transition-all", iconMode === "upload" ? "bg-background shadow-sm" : "opacity-50")}
                >
                  <Upload size={12} /> UPLOAD
                </button>
              </div>
            </div>

            {iconMode === "link" ? (
              <div className="relative">
                <ImageIcon className="absolute left-4 top-1/2 -translate-y-1/2 text-muted-foreground" size={18} />
                <input
                  type="text"
                  value={formData.icon}
                  onChange={(e) => setFormData({ ...formData, icon: e.target.value })}
                  placeholder="https://logo.com/image.png"
                  className="w-full pl-12 pr-4 py-4 bg-secondary/40 border-none rounded-2xl text-sm outline-none"
                />
              </div>
            ) : (
                <div
                  onClick={() => fileInputRef.current?.click()}
                  className="border-2 border-dashed border-border rounded-3xl p-8 flex flex-col items-center justify-center hover:bg-secondary/20 transition-colors cursor-pointer group"
                >
                  {previewUrl ? (
                    <img src={previewUrl} alt="Preview" className="w-12 h-12 rounded-xl object-cover mb-2" />
                  ) : (
                      <div className="w-12 h-12 bg-secondary rounded-2xl flex items-center justify-center mb-3 group-hover:scale-110 transition-transform">
                        <Upload className="text-muted-foreground" size={20} />
                      </div>
                    )}
                  <p className="text-xs font-bold text-muted-foreground">
                    {selectedFile ? selectedFile.name : "Click to upload SVG or PNG"}
                  </p>
                  <p className="text-[10px] text-muted-foreground/60 mt-1">Max size: 2MB</p>
                </div>
              )}
          </div>

          {/* ... rest of the input fields ... */}
          <div className="">
            {[
              { label: "Callback Endpoint", key: "callbackUrl", type: "text" },
            ].map((field) => (
              <div key={field.key} className="space-y-2">
                <label className="text-[10px] font-bold text-muted-foreground ml-1 uppercase">{field.label}</label>
                <div className="flex gap-2">
                  <div className="relative flex-1">
                    <input
                      type={field.type === "password" && !showSecret ? "password" : "text"}
                      value={(formData as any)[field.key]}
                      onChange={(e) => setFormData({ ...formData, [field.key]: e.target.value })}
                      className="w-full px-5 py-4 bg-secondary/40 border-none rounded-2xl text-sm outline-none"
                      placeholder="https://api.yoursite.com/v1/callback"
                    />
                    {field.type === "password" && (
                      <button
                        onClick={() => setShowSecret(!showSecret)}
                        className="absolute right-4 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                      >
                        {showSecret ? <EyeOff size={18} /> : <Eye size={18} />}
                      </button>
                    )}
                  </div>
                  {/* <button
                    onClick={() => copyToClipboard((formData as any)[field.key])}
                    className="p-4 bg-background border border-border rounded-2xl hover:bg-secondary transition-all active:scale-95 shadow-sm"
                  >
                    <Copy size={18} />
                  </button> */}
                  <button
                    onClick={() => copyToClipboard((formData as any)[field.key], field.key)}
                    className={cn(
                      "p-4 border rounded-2xl transition-all active:scale-95 shadow-sm",
                      copiedKey === field.key
                        ? "bg-emerald-500/10 border-emerald-500/50 text-emerald-600"
                        : "bg-background border-border hover:bg-secondary"
                    )}
                  >
                    {copiedKey === field.key ? <Check size={18} /> : <Copy size={18} />}
                  </button>
                </div>
              </div>
            ))}
          </div>

          {/* Credentials Area */}
          {initialData && <div className="bg-secondary/20 rounded-[2.5rem] p-6 md:p-8 border border-border space-y-6">
            <div className="flex items-center gap-2 text-primary/80 mb-2">
              <ShieldAlert size={18} />
              <span className="text-xs font-bold uppercase tracking-widest">Secret Credentials</span>
            </div>

            {[
              { label: "Client ID", key: "clientId", type: "text" },
              { label: "Client Secret", key: "clientSecret", type: "password" },
              { label: "Redirect Endpoint", key: "redirectUri", type: "text" }
            ].map((field) => (
              <div key={field.key} className="space-y-2">
                <label className="text-[10px] font-bold text-muted-foreground ml-1 uppercase">{field.label}</label>
                <div className="flex gap-2">
                  <div className="relative flex-1">
                    <input
                      type={field.type === "password" && !showSecret ? "password" : "text"}
                      value={(formData as any)[field.key]}
                      onChange={(e) => setFormData({ ...formData, [field.key]: e.target.value })}
                      className={cn("w-full bg-background px-5 py-4  rounded-2xl text-xs font-mono border border-border outline-none focus:ring-2 focus:ring-primary/10",
                        field.type === "password" && "pr-[50px]"
                      )

                      }
                    />
                    {field.type === "password" && (
                      <button
                        onClick={() => setShowSecret(!showSecret)}
                        className="absolute right-4 top-1/2 -translate-y-1/2 text-muted-foreground hover:text-foreground"
                      >
                        {showSecret ? <EyeOff size={18} /> : <Eye size={18} />}
                      </button>
                    )}
                  </div>

                  {field.key === "clientSecret" && (
                    <button
                      type="button"
                      onClick={() => setIsRotateConfirmOpen(true)}
                      className="p-4 bg-background border border-border rounded-2xl hover:bg-amber-500/10 hover:text-amber-600 transition-all active:scale-95 shadow-sm group"
                      title="Rotate Secret"
                    >
                      <Power size={18} className="rotate-90 group-hover:rotate-180 transition-transform duration-500" />
                    </button>
                  )}
                  <button
                    onClick={() => copyToClipboard((formData as any)[field.key], field.key)}
                    className={cn(
                      "p-4 border rounded-2xl transition-all active:scale-95 shadow-sm",
                      copiedKey === field.key
                        ? "bg-emerald-500/10 border-emerald-500/50 text-emerald-600"
                        : "bg-background border-border hover:bg-secondary"
                    )}
                  >
                    {copiedKey === field.key ? <Check size={18} /> : <Copy size={18} />}
                  </button>
                </div>
              </div>
            ))}
          </div>}
        </div>



        {/* --- Footer --- */}
        <div className="p-6 md:p-10 border-t border-border bg-secondary/10 flex flex-col md:flex-row justify-end gap-4">
          <button
            onClick={handleClose}
            disabled={isSubmitting}
            className="px-8 py-4 rounded-2xl font-bold text-sm text-muted-foreground hover:bg-secondary transition-all disabled:opacity-50"
          >
            Cancel
          </button>
          <button
            className="bg-primary text-primary-foreground px-10 py-4 rounded-2xl font-bold shadow-xl shadow-primary/20 hover:opacity-90 flex items-center justify-center gap-3 transition-all active:scale-[0.98] disabled:opacity-70"
            onClick={onFormSubmit}
            disabled={isSubmitting}
          >
            {isSubmitting ? (
              <div className="h-5 w-5 border-2 border-white/30 border-t-white rounded-full animate-spin" />
            ) : (
                <Check size={20} />
              )}
            {initialData ? "Update Integration" : "Complete Installation"}
          </button>
        </div>
      </div>

      {isDeleteConfirmOpen && (
        <div className="fixed inset-0 z-[60] flex items-center justify-center bg-black/60 backdrop-blur-sm animate-in fade-in duration-200">
          <div className="bg-background border border-border w-[90%] max-w-sm rounded-[2rem] p-8 shadow-2xl scale-in-95 animate-in zoom-in-95 duration-200">
            <div className="flex flex-col items-center text-center space-y-4">
              <div className="p-4 bg-destructive/10 text-destructive rounded-full">
                <Trash2 size={32} />
              </div>

              <div>
                <h3 className="text-xl font-bold">Uninstall App?</h3>
                <p className="text-sm text-muted-foreground mt-2">
                  This will remove the integration and all associated settings. This action is permanent.
          </p>
              </div>

              <div className="flex flex-col w-full gap-2 pt-4">
                <button
                  onClick={handleConfirmDelete}
                  disabled={deleteStatus !== "idle"}
                  className={cn(
                    "w-full py-4 rounded-2xl font-bold transition-all flex items-center justify-center gap-2",
                    deleteStatus === "success"
                      ? "bg-emerald-500 text-white"
                      : "bg-destructive text-destructive-foreground hover:opacity-90"
                  )}
                >
                  {deleteStatus === "loading" && (
                    <div className="h-4 w-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                  )}
                  {deleteStatus === "success" && <Check size={18} />}
                  {deleteStatus === "idle" && "Delete"}
                  {deleteStatus === "loading" && "Deleting..."}
                  {deleteStatus === "success" && "Deleted"}
                </button>

                <button
                  onClick={() => setIsDeleteConfirmOpen(false)}
                  disabled={deleteStatus !== "idle"}
                  className="w-full py-4 rounded-2xl font-bold text-muted-foreground hover:bg-secondary transition-all"
                >
                  Cancel
          </button>
              </div>
            </div>
          </div>
        </div>
      )}

      {isRotateConfirmOpen && (
        <div className="fixed inset-0 z-[60] flex items-center justify-center bg-black/60 backdrop-blur-sm animate-in fade-in duration-200">
          <div className="bg-background border border-border w-[90%] max-w-sm rounded-[2rem] p-8 shadow-2xl animate-in zoom-in-95 duration-200">
            <div className="flex flex-col items-center text-center space-y-4">
              <div className="p-4 bg-amber-500/10 text-amber-600 rounded-full">
                <ShieldAlert size={32} />
              </div>

              <div>
                <h3 className="text-xl font-bold">Rotate Client Secret?</h3>
                <p className="text-sm text-muted-foreground mt-2">
                  The current secret will stop working immediately. You will need to update your application code.
          </p>
              </div>

              <div className="flex flex-col w-full gap-2 pt-4">
                <button
                  onClick={handleRotateSecret}
                  disabled={rotateStatus !== "idle"}
                  className={cn(
                    "w-full py-4 rounded-2xl font-bold transition-all flex items-center justify-center gap-2",
                    rotateStatus === "success"
                      ? "bg-emerald-500 text-white"
                      : "bg-amber-600 text-white hover:bg-amber-700"
                  )}
                >
                  {rotateStatus === "loading" && (
                    <div className="h-4 w-4 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                  )}
                  {rotateStatus === "success" ? <Check size={18} /> : null}
                  {rotateStatus === "idle" && "Rotate Secret"}
                  {rotateStatus === "loading" && "Rotating..."}
                  {rotateStatus === "success" && "Rotated!"}
                </button>

                <button
                  onClick={() => setIsRotateConfirmOpen(false)}
                  disabled={rotateStatus !== "idle"}
                  className="w-full py-4 rounded-2xl font-bold text-muted-foreground hover:bg-secondary transition-all"
                >
                  Cancel
          </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
