"use client";
import React, { useState, useCallback, useEffect } from "react";
import Cropper from "react-easy-crop";
import { X, Upload, Trash2, Image as ImageIcon, Loader2, RefreshCw } from "lucide-react";
import { getCroppedImg } from "@/lib/cropImage";
import { cn } from "@/lib/utils";
import { authService } from "@/lib/auth"
import { Toast } from '@/components/ui/Toast';

interface ProfileUploadModalProps {
    isOpen: boolean;
    onClose: () => void;
    onUploadSuccess: () => void;
    currentImageUrl?: string | null;
}

export default function ProfileUploadModal({
    isOpen,
    onClose,
    onUploadSuccess,
    currentImageUrl,
    Eric }) {
    const [image, setImage] = useState<string | null>(null);
    const [crop, setCrop] = useState({ x: 0, y: 0 });
    const [zoom, setZoom] = useState(1);
    const [croppedAreaPixels, setCroppedAreaPixels] = useState(null);
    const [isProcessing, setIsProcessing] = useState(false);
    const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);
    // Sync state when modal opens
    useEffect(() => {
        if (isOpen) {
            setImage(currentImageUrl || null);
        }
    }, [isOpen, currentImageUrl]);

    const onCropComplete = useCallback((_area: any, pixels: any) => {
        setCroppedAreaPixels(pixels);
    }, []);

    const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        if (e.target.files && e.target.files.length > 0) {
            const reader = new FileReader();
            reader.onload = () => setImage(reader.result as string);
            reader.readAsDataURL(e.target.files[0]);
        }
    };


    const handleSave = async () => {
        try {
            setIsProcessing(true);

            // 1. Get the blob from your cropper
            const croppedImageBlob = await getCroppedImg(image!, croppedAreaPixels);

            // 2. Prepare FormData
            const formData = new FormData();
            formData.append("profile_picture", croppedImageBlob, "profile.jpg");

            // 3. Use authService (automatically handles refresh & headers)
            const result = await authService.uploadProfilePicture(formData);

            if (result.success) {
                onUploadSuccess(); // Refresh user data in parent
                onClose();
            } else {
                // console.error("Upload failed:", result.message);
                setToast({ type: "error", message: result.message });
            }
        } catch (error) {
            // console.error("Upload process error:", error);
            setToast({ type: "error", message: "Internal server error" });
        } finally {
            setIsProcessing(false);
        }
    };

    const handleRemovePhoto = async () => {
        try {
            setIsProcessing(true);

            // Use authService method
            const result = await authService.removeProfilePicture();

            if (result.success) {
                onUploadSuccess();
                onClose();
            } else {
                console.error("Remove failed:", result.message);
            }
        } catch (e) {
            console.error("Remove error:", e);
        } finally {
            setIsProcessing(false);
        }
    };

    // const handleSave = async () => {
    //     try {
    //         setIsProcessing(true);
    //         const croppedImageBlob = await getCroppedImg(image!, croppedAreaPixels);

    //         const formData = new FormData();
    //         formData.append("profile_picture", croppedImageBlob, "profile.jpg");

    //         const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/profile/upload-picture/`, {
    //             method: "POST",
    //             body: formData,
    //             credentials: "include",
    //         });

    //         if (response.ok) {
    //             onUploadSuccess();
    //             onClose();
    //         }
    //     } catch (error) {
    //         console.error("Upload failed", error);
    //     } finally {
    //         setIsProcessing(false);
    //     }
    // };

    // const handleRemovePhoto = async () => {
    //     try {
    //         setIsProcessing(true);
    //         const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/profile/remove-picture/`, {
    //             method: "DELETE",
    //             credentials: "include",
    //         });
    //         if (response.ok) {
    //             onUploadSuccess();
    //             onClose();
    //         }
    //     } catch (e) {
    //         console.error(e);
    //     } finally {
    //         setIsProcessing(false);
    //     }
    // };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 z-[100] flex items-center justify-center p-4 bg-black/60 backdrop-blur-sm animate-in fade-in duration-200">
            <div className="relative bg-card border w-full max-w-md rounded-3xl shadow-2xl overflow-hidden animate-in zoom-in-95 duration-200">

                {/* Header */}
                <div className="p-5 border-b flex justify-between items-center">
                    <h3 className="text-lg font-bold">Profile Picture</h3>
                    <button onClick={onClose} className="p-2 hover:bg-muted rounded-full transition-colors">
                        <X size={20} />
                    </button>
                </div>

                <div className="p-6">
                    {!image ? (
                        /* Upload State */
                        <label className="flex flex-col items-center justify-center w-full h-72 border-2 border-dashed border-muted rounded-2xl cursor-pointer hover:bg-accent/50 hover:border-sidebar-primary/50 transition-all group">
                            <div className="p-4 rounded-full bg-sidebar-primary/10 text-sidebar-primary mb-4 group-hover:scale-110 transition-transform">
                                <Upload size={32} />
                            </div>
                            <p className="font-semibold text-sm">Upload new photo</p>
                            <p className="text-xs text-muted-foreground mt-1">JPG or PNG accepted</p>
                            <input type="file" className="hidden" accept="image/png, image/jpeg" onChange={handleFileChange} />
                        </label>
                    ) : (
                            /* Edit/Crop State */
                            <div className="space-y-6">
                                <div className="relative w-full h-72 bg-neutral-900 rounded-2xl overflow-hidden shadow-inner">
                                    <Cropper
                                        image={image}
                                        crop={crop}
                                        zoom={zoom}
                                        aspect={1}
                                        onCropChange={setCrop}
                                        onCropComplete={onCropComplete}
                                        onZoomChange={setZoom}
                                    />
                                </div>

                                <div className="flex flex-col gap-4">
                                    <div className="space-y-2">
                                        <div className="flex justify-between text-[10px] font-bold uppercase text-muted-foreground">
                                            <span>Zoom Control</span>
                                            <span>{Math.round(zoom * 100)}%</span>
                                        </div>
                                        <input
                                            type="range"
                                            min={1} max={3} step={0.1}
                                            value={zoom}
                                            onChange={(e) => setZoom(Number(e.target.value))}
                                            className="w-full h-1.5 bg-muted rounded-lg appearance-none cursor-pointer accent-sidebar-primary"
                                        />
                                    </div>

                                    {/* Sub-Actions */}
                                    <div className="flex gap-2">
                                        <label className="flex-1 flex items-center justify-center gap-2 px-3 py-2.5 bg-secondary hover:bg-secondary/80 rounded-xl text-xs font-bold cursor-pointer transition-colors">
                                            <RefreshCw size={14} />
                    Change Image
                    <input type="file" className="hidden" accept="image/png, image/jpeg" onChange={handleFileChange} />
                                        </label>

                                        {currentImageUrl && (
                                            <button
                                                onClick={handleRemovePhoto}
                                                className="flex items-center justify-center gap-2 px-4 py-2.5 bg-destructive/10 hover:bg-destructive/20 text-destructive rounded-xl text-xs font-bold transition-colors"
                                            >
                                                <Trash2 size={14} />
                      Remove
                                            </button>
                                        )}
                                    </div>
                                </div>
                            </div>
                        )}
                </div>

                {/* Footer */}
                <div className="p-5 bg-muted/30 border-t flex gap-3">
                    <button
                        onClick={onClose}
                        className="flex-1 px-4 py-3 text-sm font-bold text-muted-foreground hover:bg-muted rounded-xl transition-colors"
                    >
                        Cancel
          </button>
                    <button
                        disabled={!image || isProcessing}
                        onClick={handleSave}
                        className="flex-[2] bg-sidebar-primary text-white px-4 py-3 rounded-xl text-sm font-bold disabled:opacity-50 flex items-center justify-center gap-2 shadow-lg shadow-sidebar-primary/20 active:scale-[0.98] transition-all"
                    >
                        {isProcessing ? <Loader2 className="animate-spin" size={18} /> : "Save Changes"}
                    </button>
                </div>
            </div>

            {toast && (
                <Toast
                    message={toast.message}
                    type={toast.type}
                    onClose={() => setToast(null)}
                />
            )}
        </div>
    );
}

// "use client";
// import React, { useState, useCallback } from "react";
// import Cropper from "react-easy-crop";
// import { X, Upload, Camera, Loader2 } from "lucide-react";
// import { getCroppedImg } from "@/lib/cropImage"; // Path to utility above
// import { cn } from "@/lib/utils";

// interface ProfileUploadModalProps {
//   isOpen: boolean;
//   onClose: () => void;
//   onUploadSuccess: () => void;
// }

// export default function ProfileUploadModal({ isOpen, onClose, onUploadSuccess }: ProfileUploadModalProps) {
//   const [image, setImage] = useState<string | null>(null);
//   const [crop, setCrop] = useState({ x: 0, y: 0 });
//   const [zoom, setZoom] = useState(1);
//   const [croppedAreaPixels, setCroppedAreaPixels] = useState(null);
//   const [isUploading, setIsUploading] = useState(false);

//   const onCropComplete = useCallback((_area: any, pixels: any) => {
//     setCroppedAreaPixels(pixels);
//   }, []);

//   const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
//     if (e.target.files && e.target.files.length > 0) {
//       const reader = new FileReader();
//       reader.addEventListener("load", () => setImage(reader.result as string));
//       reader.readAsDataURL(e.target.files[0]);
//     }
//   };

//   const handleSave = async () => {
//     try {
//       setIsUploading(true);
//       const croppedImageBlob = await getCroppedImg(image!, croppedAreaPixels);

//       const formData = new FormData();
//       formData.append("profile_picture", croppedImageBlob, "profile.jpg");

//       const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/profile/upload-picture/`, {
//         method: "POST",
//         body: formData,
//         credentials: "include",
//       });

//       const data = await response.json()
//       console.log(data)
//       if (response.ok) {
//         onUploadSuccess();
//         handleClose();
//       }
//     } catch (error) {
//       console.error("Upload failed", error);
//     } finally {
//       setIsUploading(false);
//     }
//   };

//   const handleClose = () => {
//     setImage(null);
//     onClose();
//   };

//   if (!isOpen) return null;

//   return (
//     <div className="fixed inset-0 z-[100] flex items-center justify-center p-4">
//       <div className="absolute inset-0 bg-background/80 backdrop-blur-sm" onClick={handleClose} />

//       <div className="relative bg-card border w-full max-w-md rounded-2xl shadow-2xl overflow-hidden animate-in zoom-in-95 duration-200">
//         <div className="p-4 border-b flex justify-between items-center">
//           <h3 className="font-bold">Update Profile Picture</h3>
//           <button onClick={handleClose} className="p-1 hover:bg-muted rounded-full"><X size={20} /></button>
//         </div>

//         <div className="p-6">
//           {!image ? (
//             <label className="flex flex-col items-center justify-center w-full h-64 border-2 border-dashed border-muted rounded-2xl cursor-pointer hover:bg-accent/50 transition-colors group">
//               <div className="flex flex-col items-center justify-center pt-5 pb-6">
//                 <div className="p-4 rounded-full bg-sidebar-primary/10 text-sidebar-primary mb-4 group-hover:scale-110 transition-transform">
//                   <Upload size={32} />
//                 </div>
//                 <p className="text-sm font-medium">Click to upload or drag and drop</p>
//                 <p className="text-xs text-muted-foreground mt-1">JPG or PNG (max 5MB)</p>
//               </div>
//               <input type="file" className="hidden" accept="image/png, image/jpeg" onChange={handleFileChange} />
//             </label>
//           ) : (
//             <div className="space-y-6">
//               <div className="relative w-full h-72 bg-black rounded-xl overflow-hidden">
//                 <Cropper
//                   image={image}
//                   crop={crop}
//                   zoom={zoom}
//                   aspect={1}
//                   onCropChange={setCrop}
//                   onCropComplete={onCropComplete}
//                   onZoomChange={setZoom}
//                 />
//               </div>

//               <div className="space-y-2">
//                 <label className="text-xs font-bold text-muted-foreground uppercase">Zoom</label>
//                 <input
//                   type="range"
//                   min={1}
//                   max={3}
//                   step={0.1}
//                   value={zoom}
//                   onChange={(e) => setZoom(Number(e.target.value))}
//                   className="w-full h-1.5 bg-muted rounded-lg appearance-none cursor-pointer accent-sidebar-primary"
//                 />
//               </div>
//             </div>
//           )}
//         </div>

//         <div className="p-4 bg-muted/30 border-t flex justify-end gap-3">
//           <button onClick={handleClose} className="px-4 py-2 text-sm font-semibold">Cancel</button>
//           <button 
//             disabled={!image || isUploading}
//             onClick={handleSave}
//             className="bg-sidebar-primary text-white px-6 py-2 rounded-xl text-sm font-bold disabled:opacity-50 flex items-center gap-2"
//           >
//             {isUploading ? <Loader2 className="animate-spin" size={16} /> : "Save Picture"}
//           </button>
//         </div>
//       </div>
//     </div>
//   );
// }