"use client";
import { useState, useEffect } from "react";
import {
    Info, X, Clock, Shield, Terminal, User, Mail,
    Phone, Activity, Fingerprint, ShieldAlert
} from "lucide-react";
import { cn } from "@/lib/utils";
import { adminAuthService } from '@/lib/adminAuthService';

export default function LogDetailModal({ log, isOpen, onClose }: { log: any, isOpen: boolean, onClose: () => void }) {
    const [userInfo, setUserInfo] = useState<any>(null);
    const [loadingUser, setLoadingUser] = useState(false);
    const [fullLog, setFullLog] = useState<any>(null);
    const [isLoading, setIsLoading] = useState(false);

    const [message, setMessage] = useState<string | null>(null);


    useEffect(() => {
        if (isOpen && log.log_id) {
            loadLogDetail(log.log_id);
        } else {
            setFullLog(null);
        }
    }, [isOpen, log]);

    const loadLogDetail = async (id: string) => {
        setIsLoading(true);
        setLoadingUser(true);
        try {
            const result = await adminAuthService.getLogDetail(id);

            if (result.success) {
                setFullLog(result.data);
                setUserInfo(result.data);
            } else {
                setMessage(result.message || "Could not retrieve log details")
            }
        } catch (err) {
            setMessage("Network error occurred")
        } finally {
            setIsLoading(false);
            setLoadingUser(false)
        }
    };


    const handleOnClose = () => {
        setUserInfo(null);
        setLoadingUser(false);
        setFullLog(null);
        setIsLoading(false);
        setMessage(null)
        onClose()
    }

    if (!isOpen || !log) return null;

    return (
        <div className="fixed inset-0 z-[100] flex items-center justify-center p-4">
            <div className="absolute inset-0 bg-background/80 backdrop-blur-sm" onClick={handleOnClose} />

            <div className="relative w-full max-w-2xl bg-background border border-border rounded-[32px] shadow-2xl overflow-hidden animate-in fade-in zoom-in duration-200">

                {/* Header */}
                <div className="p-6 border-b border-border flex items-center justify-between bg-secondary/5">
                    <div className="flex items-center gap-3">
                        <div className="p-2.5 bg-primary/10 text-primary rounded-xl">
                            <Terminal size={20} />
                        </div>
                        <div>
                            <h2 className="text-xl font-bold">Log Deep-Dive</h2>
                            <p className="text-xs text-muted-foreground font-mono">Trace ID: {log.log_id || 'N/A'}</p>
                        </div>
                    </div>
                    <button onClick={handleOnClose} className="p-2 hover:bg-secondary rounded-full transition-colors"><X size={20} /></button>
                </div>




                <div className="p-6 max-h-[75vh] overflow-y-auto space-y-8">

                    {message && (
                        <div className={cn(
                            "flex items-start gap-3 p-4 rounded-2xl animate-in fade-in slide-in-from-top-2 mb-6",
                            "bg-destructive/10 text-red-600 border border-destructive/20",

                        )}>
                            <ShieldAlert size={18} className="mt-0.5 shrink-0" />
                            <div className="flex-1 text-sm font-medium">
                                {message}
                            </div>
                            <button
                                onClick={() => setMessage(null)}
                                className="opacity-70 hover:opacity-100 transition-opacity"
                            >
                                <X size={16} />
                            </button>
                        </div>
                    )}


                    {/* Top Section: Quick Info */}
                    <div className="grid grid-cols-2 gap-4">
                        <div className="space-y-1">
                            <span className="text-[10px] font-bold text-muted-foreground uppercase tracking-widest flex items-center gap-1">
                                <Clock size={10} /> Timestamp
              </span>
                            <p className="text-sm font-medium">{new Date(log.created_at).toLocaleString()}</p>
                        </div>
                        <div className="space-y-1 text-right">
                            <span className="text-[10px] font-bold text-muted-foreground uppercase tracking-widest flex items-center gap-1 justify-end">
                                <Shield size={10} /> Action Type
              </span>
                            <div className="inline-block px-2.5 py-1 rounded-lg bg-primary/5 text-primary text-[10px] font-bold border border-primary/10">
                                {log.action}
                            </div>
                        </div>
                    </div>


                    {/* Section 1: Performer Info (With Skeleton) */}
                    <section className="space-y-4">
                        <h4 className="text-[10px] font-bold text-muted-foreground uppercase tracking-[0.2em] flex items-center gap-2">
                            <User size={12} /> Performed By
            </h4>

                        <div className="p-5 border border-border rounded-2xl bg-secondary/10 relative overflow-hidden">
                            {loadingUser ? (
                                <div className="space-y-3 animate-pulse">
                                    <div className="h-5 w-40 bg-foreground/10 rounded-lg" />
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="h-4 w-full bg-foreground/5 rounded-md" />
                                        <div className="h-4 w-3/4 bg-foreground/5 rounded-md" />
                                    </div>
                                </div>
                            ) : (
                                    <div className="space-y-4">
                                        <div className="flex items-center justify-between">
                                            <h3 className="text-lg font-bold">{userInfo?.first_name}</h3>
                                            <div className="flex gap-2">

                                                <StatusBadge active={userInfo?.is_active} />
                                                {/* <StatusBadge active={userInfo?.is_verified} /> */}
                                            </div>
                                        </div>
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-y-3 gap-x-6 text-sm">
                                            <div className="flex items-center gap-2 text-muted-foreground">
                                                <Mail size={14} /> {userInfo?.primary_email}
                                            </div>
                                            <div className="flex items-center gap-2 text-muted-foreground">
                                                <Phone size={14} /> {userInfo?.phone_number}
                                            </div>
                                        </div>
                                    </div>
                                )}
                        </div>
                    </section>

                    {/* Section 2: Metadata List */}
                    <section className="space-y-4">
                        <h4 className="text-[10px] font-bold text-muted-foreground uppercase tracking-[0.2em] flex items-center gap-2">
                            <Fingerprint size={12} /> Extended Metadata
            </h4>
                        <div className="bg-secondary/5 border border-border rounded-2xl divide-y divide-border/40">
                            {log.metadata ? Object.entries(log.metadata).map(([key, value]) => (
                                <div key={key} className="p-4 flex flex-col gap-1.5 hover:bg-secondary/20 transition-colors">
                                    <span className="text-[10px] font-bold text-primary/60 uppercase">{key.replace(/_/g, ' ')}</span>
                                    <div className="text-sm font-mono break-all leading-relaxed">
                                        {typeof value === 'object' ? JSON.stringify(value, null, 2) : String(value)}
                                    </div>
                                </div>
                            )) : (
                                    <div className="p-8 text-center text-sm text-muted-foreground italic">No metadata found</div>
                                )}
                        </div>
                    </section>
                </div>

                <div className="p-4 bg-secondary/10 border-t border-border flex justify-end">
                    <button onClick={handleOnClose} className="px-8 py-2.5 bg-primary text-white rounded-xl text-sm font-bold hover:opacity-90 transition-all">
                        Close
          </button>
                </div>
            </div>
        </div>
    );
}

// Helper Badge
const StatusBadge = ({ active }: { active: boolean }) => (
    <div className={cn(
        "px-2 py-0.5 rounded-md text-[10px] font-black uppercase border",
        active ? "bg-emerald-500/10 text-emerald-600 border-emerald-500/20" : "bg-rose-500/10 text-rose-600 border-rose-500/20"
    )}>
        {active ? "Active" : "Inactive"}
    </div>
);

// "use client";
// import { cn } from "@/lib/utils";

// // Define the available status types for better Type Safety
// export type StatusType = 'active' | 'inactive' | 'verified' | 'unverified' | 'suspended' | 'pending';

// interface StatusBadgeProps {
//     status: StatusType | string;
//     className?: string;
// }

// const StatusBadge = ({ status, className }: StatusBadgeProps) => {
//     // Map status keys to specific styles
//     const styles: Record<string, string> = {
//         active: "bg-emerald-500/10 text-emerald-600 border-emerald-500/20",
//         verified: "bg-blue-500/10 text-blue-600 border-blue-500/20",
//         inactive: "bg-slate-500/10 text-slate-600 border-slate-500/20",
//         unverified: "bg-amber-500/10 text-amber-600 border-amber-500/20",
//         suspended: "bg-rose-500/10 text-rose-600 border-rose-500/20",
//         pending: "bg-indigo-500/10 text-indigo-600 border-indigo-500/20",
//     };

//     // Fallback style if status doesn't match
//     const currentStyle = styles[status.toLowerCase()] || styles.inactive;

//     return (
//         <div className={cn(
//             "px-2.5 py-0.5 rounded-lg text-[10px] font-black uppercase border tracking-wider inline-flex items-center justify-center",
//             currentStyle,
//             className
//         )}>
//             {status.replace(/_/g, ' ')}
//         </div>
//     );
// };

// "use client";
// import { Info, X, Clock, User, Shield, Terminal, Copy } from "lucide-react";
// import { cn } from "@/lib/utils";

// interface LogDetailModalProps {
//   log: any;
//   isOpen: boolean;
//   onClose: () => void;
// }

// export default function LogDetailModal({ log, isOpen, onClose }: LogDetailModalProps) {
//   if (!isOpen || !log) return null;
//   console.log(log)

//   // Helper to render metadata dynamically
//   const renderMetadata = (data: any) => {
//     if (!data || typeof data !== "object") return <p className="text-muted-foreground italic text-xs">No additional metadata</p>;

//     return Object.entries(data).map(([key, value]) => (
//       <div key={key} className="group flex flex-col gap-1 py-2 border-b border-border/50 last:border-0">
//         <span className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground/70">{key.replace(/_/g, ' ')}</span>
//         <div className="text-sm font-mono text-foreground break-all bg-secondary/30 p-2 rounded-lg">
//           {typeof value === 'object' ? JSON.stringify(value, null, 2) : String(value)}
//         </div>
//       </div>
//     ));
//   };

//   return (
//     <div className="fixed inset-0 z-[100] flex items-center justify-center p-4">
//       {/* Backdrop */}
//       <div className="absolute inset-0 bg-background/80 backdrop-blur-sm" onClick={onClose} />

//       {/* Modal Content */}
//       <div className="relative w-full max-w-2xl bg-background border border-border rounded-[32px] shadow-2xl overflow-hidden animate-in fade-in zoom-in duration-200">

//         {/* Header */}
//         <div className="p-6 border-b border-border flex items-center justify-between bg-secondary/10">
//           <div className="flex items-center gap-3">
//             <div className="p-2.5 bg-primary/10 text-primary rounded-xl">
//               <Terminal size={20} />
//             </div>
//             <div>
//               <h2 className="text-xl font-bold">Event Details</h2>
//               <p className="text-xs text-muted-foreground font-mono">Log Id : {log.log_id || 'System Action'}</p>
//             </div>
//           </div>
//           <button onClick={onClose} className="p-2 hover:bg-secondary rounded-full transition-colors">
//             <X size={20} />
//           </button>
//         </div>

//         {/* Body */}
//         <div className="p-6 max-h-[70vh] overflow-y-auto space-y-8">

        //   {/* Top Section: Quick Info */}
        //   <div className="grid grid-cols-2 gap-4">
        //     <div className="space-y-1">
        //       <span className="text-[10px] font-bold text-muted-foreground uppercase tracking-widest flex items-center gap-1">
        //         <Clock size={10} /> Timestamp
        //       </span>
        //       <p className="text-sm font-medium">{new Date(log.created_at).toLocaleString()}</p>
        //     </div>
        //     <div className="space-y-1 text-right">
        //       <span className="text-[10px] font-bold text-muted-foreground uppercase tracking-widest flex items-center gap-1 justify-end">
        //         <Shield size={10} /> Action Type
        //       </span>
        //       <div className="inline-block px-2.5 py-1 rounded-lg bg-primary/5 text-primary text-[10px] font-bold border border-primary/10">
        //         {log.action}
        //       </div>
        //     </div>
        //   </div>

//           <hr className="border-border" />

//           {/* Dynamic Metadata Section */}
//           <div className="space-y-4">
//             <h4 className="text-sm font-bold flex items-center gap-2">
//               <Info size={16} className="text-primary" />
//               Event Metadata
//             </h4>
//             <div className="bg-secondary/10 border border-border rounded-2xl p-4 space-y-1">
//               {renderMetadata(log.metadata)}
//             </div>
//           </div>
//         </div>

//         {/* Footer */}
//         <div className="p-4 bg-secondary/20 border-t border-border flex justify-end gap-3">
//           <button 
//             onClick={onClose}
//             className="px-6 py-2.5 bg-background border border-border rounded-xl text-sm font-semibold hover:bg-secondary transition-all"
//           >
//             Close
//           </button>
//         </div>
//       </div>
//     </div>
//   );
// }