"use client";
import { useState, useEffect, useCallback, useMemo } from "react";
import {
    AreaChart, Area, XAxis, YAxis, CartesianGrid,
    Tooltip, ResponsiveContainer, Loader2, ChevronDown, Calendar
} from "lucide-react";
import {
    AreaChart as ReAreaChart,
    Area as ReArea,
    XAxis as ReXAxis,
    YAxis as ReYAxis,
    CartesianGrid as ReGrid,
    Tooltip as ReTooltip,
    ResponsiveContainer as ReContainer
} from "recharts";
import { cn } from "@/lib/utils";
import { adminAuthService } from '@/lib/adminAuthService';
import { Toast } from '@/components/ui/Toast';

const API_BASE = process.env.NEXT_PUBLIC_API_BASE_URL || "http://localhost:8000";

export default function UserTrendsChart() {
    const [data, setData] = useState([]);
    const [loading, setLoading] = useState(true);
    const [view, setView] = useState("7"); // "7", "30", or "custom"
    const [isDropdownOpen, setIsDropdownOpen] = useState(false);
    const [range, setRange] = useState({ from: "", to: "" });
    // Get today's date in YYYY-MM-DD format for the 'max' attribute
    const today = useMemo(() => new Date().toISOString().split("T")[0], []);
    const [toast, setToast] = useState<{ type: 'success' | 'error', message: string } | null>(null);

    
    // const fetchData = useCallback(async () => {
    //     setLoading(true);
    //     try {
    //         let url = `${API_BASE}/api/admin/dashboard/trends/`;
    //         if (view === "custom") {
    //             if (range.from && range.to) {
    //                 url += `?from=${range.from}&to=${range.to}`;
    //             } else {
    //                 setLoading(false);
    //                 return; // Don't fetch until both dates are selected
    //             }
    //         } else {
    //             url += `?days=${view}`;
    //         }
    //         const res = await fetch(url, { credentials: "include" });
    //         const json = await res.json();
    //         setData(json.trends || []);
    //     } catch (err) {
    //         console.error("Fetch error:", err);
    //     } finally {
    //         setLoading(false);
    //     }
    // }, [view, range]);


    const fetchData = useCallback(async () => {
        // Validation logic stays in the component
        if (view === "custom" && (!range.from || !range.to)) {
            return;
        }

        setLoading(true);

        // 1. Call the service with clean arguments
        const result = await adminAuthService.getDashboardTrends(view, range);

        // 2. Handle result and toast
        if (result.success) {
            // Accessing .trends as per your original json.trends logic
            setData(result.data?.trends || []);
        } else {
            setToast({
                type: "error",
                message: result.message || "Error fetching trend data"
            });
        }

        setLoading(false);
    }, [view, range]);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    return (
        <div className="w-full flex flex-col gap-6 bg-background">
            {/* Header with Custom Select & Date Picker */}

            <div className="flex flex-col min-[1300px]:flex-row min-[1300px]:items-center justify-between mb-8 gap-6 px-2">
                <div>
                    <h2 className="text-xl font-bold tracking-tight">System Performance</h2>
                    <p className="text-xs text-muted-foreground font-medium uppercase tracking-widest mt-1">User Engagement Metrics</p>
                </div>

                <div className="flex flex-col min-[1300px]:flex-row min-[1300px]:items-center gap-4">
                    <div className="relative">
                        <button
                            onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                            className="w-full sm:w-auto flex items-center justify-between gap-3 bg-background border border-border px-4 py-2.5 rounded-2xl text-sm font-semibold transition-all hover:bg-secondary active:scale-95"
                        >
                            <span className="flex items-center gap-2">
                                {view === "custom" ? "Custom Range" : `Last ${view} Days`}
                            </span>
                            <ChevronDown size={16} className={cn("transition-transform duration-300", isDropdownOpen && "rotate-180")} />
                        </button>

                        <div className={cn(
                            "absolute right-0 mt-2 w-48 bg-background border border-border rounded-2xl shadow-xl z-50 overflow-hidden transition-all duration-200 origin-top-right",
                            isDropdownOpen ? "opacity-100 scale-100" : "opacity-0 scale-95 pointer-events-none"
                        )}>
                            {["7", "30", "custom"].map((opt) => (
                                <button
                                    key={opt}
                                    onClick={() => { setView(opt); setIsDropdownOpen(false); }}
                                    className="w-full text-left px-5 py-3 text-sm font-medium hover:bg-primary/10 hover:text-primary transition-colors"
                                >
                                    {opt === "custom" ? "Custom Date Range" : `Last ${opt} Days`}
                                </button>
                            ))}
                        </div>
                    </div>


                    {view === "custom" && (
                        <div className="flex items-center gap-2 animate-in fade-in slide-in-from-top-2 min-[1200px]:slide-in-from-right-2 duration-300">
                            <div className="flex-1 sm:flex-none relative">
                                <input
                                    type="date"
                                    max={today}
                                    className="appearance-none w-full bg-background border border-border rounded-xl px-3 py-2 text-xs outline-none focus:ring-2 focus:ring-primary/20"
                                    onChange={(e) => setRange({ ...range, from: e.target.value })}
                                />
                            </div>
                            <span className="text-muted-foreground font-bold">-</span>
                            <div className="flex-1 sm:flex-none relative">
                                <input
                                    type="date"
                                    max={today}
                                    className="appearance-none w-full bg-background border border-border rounded-xl px-3 py-2 text-xs outline-none focus:ring-2 focus:ring-primary/20"
                                    onChange={(e) => setRange({ ...range, to: e.target.value })}
                                />
                            </div>
                        </div>
                    )}
                </div>
            </div>

            {/* Chart Canvas */}
            <div className="relative w-full h-[350px] bg-secondary/10 rounded-[2.5rem] border border-border/50 p-6 overflow-hidden">
                {loading && (
                    <div className="absolute inset-0 z-10 flex flex-col items-center justify-center bg-background/40 backdrop-blur-[2px] transition-opacity">
                        <Loader2 className="animate-spin text-primary mb-2" size={32} />
                        <span className="text-[10px] font-bold uppercase tracking-widest text-muted-foreground">Fetching Data...</span>
                    </div>
                )}

                <ReContainer width="100%" height="100%">
                    <ReAreaChart data={data} margin={{ top: 10, right: 10, left: -20, bottom: 0 }}>
                        <defs>
                            <linearGradient id="colorMetric" x1="0" y1="0" x2="0" y2="1">
                                <stop offset="5%" stopColor="hsl(var(--primary))" stopOpacity={0.3} />
                                <stop offset="95%" stopColor="hsl(var(--primary))" stopOpacity={0} />
                            </linearGradient>
                        </defs>
                        <ReGrid strokeDasharray="3 3" vertical={false} stroke="rgba(255,255,255,0.05)" />
                        <ReXAxis
                            dataKey="date"
                            axisLine={false}
                            tickLine={false}
                            tick={{ fontSize: 10, fill: '#888' }}
                            tickFormatter={(val) => new Date(val).toLocaleDateString("en-US", { month: "short", day: "numeric" })}
                        />
                        <ReYAxis axisLine={false} tickLine={false} tick={{ fontSize: 10, fill: '#888' }} />
                        <ReTooltip
                            contentStyle={{ backgroundColor: '#18181b', border: '1px solid #27272a', borderRadius: '16px', fontSize: '12px' }}
                        />

                        <ReArea
                            name="Logins"
                            type="monotone"
                            dataKey="logins"
                            stroke="#10b981"
                            strokeWidth={3}
                            fillOpacity={1}
                            fill="url(#colorLogins)"
                        />

                        <ReArea
                            name="Failed"
                            type="monotone"
                            dataKey="failed_logins"
                            stroke="#ef4444"
                            strokeWidth={2}
                            fill="transparent"
                            strokeDasharray="5 5"
                        />

                        <ReArea
                            name="New Users"
                            type="monotone"
                            dataKey="new_users"
                            stroke="#3b82f6"
                            strokeWidth={3}
                            fillOpacity={1}
                            fill="url(#colorUsers)"
                        />
                    </ReAreaChart>
                </ReContainer>
            </div>
        </div>
    );
}
