"use client";
import { useState, useEffect } from "react";
import { X, Save, Shield, Mail, Phone, Lock, Calendar, User, Info, CheckCircle2, AlertCircle, Loader2 } from "lucide-react";
import { cn } from "@/lib/utils";
import CustomCalendar from '@/components/CustomCalendar';
import { adminAuthService } from '@/lib/adminAuthService';
import CountrySelector from "@/components/CountrySelector";
import { Country } from "@/lib/country-data";

export default function EditUserModal({ user, isOpen, onClose, onSave, error }: any) {
  const [formData, setFormData] = useState<any>(null);
  const [isOpenCalendar, setIsOpenCalendar] = useState(false);
  const [editUserError, setEditUserError] = useState("")
  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleCreateUser = async (newUserData: any) => {
    setIsSubmitting(true);
    try {
      const result = await adminAuthService.createUser(newUserData);
      console.log(result)
      if (result.success) {
        handleOnClose()
      } else {
        setEditUserError(result.message || "Failed to create user")

      }
    } catch (err) {
      setEditUserError("Internal Server Error")
    } finally {
      setIsSubmitting(false);
    }
  };

  useEffect(() => {
    console.log(user)
    if (user) setFormData({
      ...user,
      country_code: user.counter_code || "91",
      recovery_phone_country_code: user.recovery_phone_country_code || "91"
    });
  }, [user]);

  if (!isOpen || !formData) return null;

  const handleChange = (field: string, value: any) => {
    setFormData((prev: any) => ({ ...prev, [field]: value }));
  };

  const handleOnClose = () => {
    setFormData(null)
    setEditUserError("")
    onClose()
  }

  return (
    <div className="fixed inset-0 z-[100] flex items-center justify-center bg-black/60 backdrop-blur-md transition-all">
      {/* Container: Full screen on mobile, max-width on desktop */}
      <div className="bg-background border-border w-full h-full sm:h-[90vh] sm:max-w-3xl sm:rounded-[2.5rem] overflow-hidden shadow-2xl flex flex-col transition-all border">

        {/* Header */}
        <div className="p-6 border-b border-border flex justify-between items-center bg-secondary/5">
          <div className="flex items-center gap-3">
            <div className="w-12 h-12 rounded-2xl bg-primary/10 flex items-center justify-center text-primary">
              <User size={24} />
            </div>
            <div>
              <h2 className="text-xl font-bold tracking-tight">User Configuration</h2>
              <p className="text-[10px] text-muted-foreground uppercase font-bold tracking-widest mt-0.5 opacity-70">
                System Reference: {formData.user_id || "N/A"}
              </p>
            </div>
          </div>
          <button onClick={handleOnClose} className="p-3 hover:bg-secondary rounded-2xl transition-all active:scale-90">
            <X size={20} />
          </button>
        </div>
        {error && (
          <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
            <AlertCircle size={16} />
            <span>{error}</span>
          </div>
        )}

        {editUserError && (
          <div className="flex items-center gap-2 text-red-600 text-sm bg-destructive/10 p-3 rounded-xl animate-in fade-in zoom-in-95">
            <AlertCircle size={16} />
            <span>{editUserError}</span>
          </div>
        )}

        {/* Scrollable Content */}
        <div className="p-4 sm:p-8 overflow-y-auto space-y-8 flex-1 scrollbar-hide">

          {/* Section: Identity */}
          <CategoryCard title="Personal Identity" icon={<User size={16} />}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-5">
              <Input label="First Name" value={formData.first_name} onChange={(v) => handleChange('first_name', v)} />
              <Input label="Last Name" value={formData.last_name} onChange={(v) => handleChange('last_name', v)} />
              <div className="flex flex-col gap-2">
                <label className="text-[11px] font-bold text-muted-foreground uppercase ml-1">Gender</label>
                <select
                  className="appearance-none bg-secondary/30 border border-border/50 rounded-2xl px-4 py-3 text-sm outline-none focus:ring-2 focus:ring-primary/20  cursor-pointer"
                  value={formData.gender || ""}
                  onChange={(e) => handleChange('gender', e.target.value)}
                >
                  <option value="">Not Specified</option>
                  <option value="male">Male</option>
                  <option value="female">Female</option>
                  <option value="other">Other</option>
                </select>
              </div>
              {/* <Input label="Date of Birth" type="date" value={formData.dob} onChange={(v) => handleChange('dob', v)} /> */}
              <div className="relative"
                onClick={() => setIsOpenCalendar(!isOpenCalendar)}
              >
                {/* <input
                  type="text"
                  className="w-full bg-transparent border-2 border-muted focus:border-sidebar-primary rounded-xl px-4 py-4 outline-none appearance-none"
                  value={formData.dob}
                  onChange={(v) => handleChange('dob', v)}
                /> */}
                <Input label="Date of Birth" value={formData.dob} onChange={(v) => handleChange('dob', v)} />
                {/* <Calendar className="absolute right-4 top-4 text-muted-foreground pointer-events-none" size={20} /> */}

                {/* <label className="absolute left-3 -top-2.5 px-2 bg-card text-xs text-sidebar-primary">Select Birthday</label> */}
              </div>
            </div>
          </CategoryCard>

          {isOpenCalendar && (
            <div className="fixed w-full h-full top-0 left-0 z-50 flex justify-center align-middle animate-in fade-in zoom-in-95 duration-500" style={{
              alignItems: 'center',
              backdropFilter: 'blur(10px)'
            }}
              onClick={() => setIsOpenCalendar(false)}

            >
              <div
                onClick={(e) => e.stopPropagation()}
                className="relative"
              >
                <CustomCalendar
                  selectedDate={formData.dob} // "2002-08-10"
                  onSelect={(newDate) => { handleChange('dob', newDate); setIsOpenCalendar(false) }}

                /></div>
            </div>
          )}

          {/* Section: Communications */}
          <CategoryCard title="Communication Channels" icon={<Mail size={16} />}>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
              <div className="relative w-full">
                {<div className="absolute z-1 left-4 top-[46px] -translate-y-1/2 flex items-center gap-2 border-r border-border pr-2 text-muted-foreground">
                  <CountrySelector onChange={(v) => { handleChange("country_code", v.dial) }} defaultValue={formData.country_code || "91"} />
                </div>}
                <Input label="Phone Number" value={formData.phone_number} onChange={(v) => handleChange('phone_number', v)} />
              </div>
              <Input label="Primary Email" value={formData.primary_email} onChange={(v) => handleChange('primary_email', v)} />
              <Input label="Recovery Email" value={formData.recovery_email} onChange={(v) => handleChange('recovery_email', v)} />
              <div className="relative w-full">
                {<div className="absolute z-1 left-4 top-[46px] -translate-y-1/2 flex items-center gap-2 border-r border-border pr-2 text-muted-foreground">
                  <CountrySelector onChange={(v) => handleChange('recovery_phone_country_code', v.dial)} defaultValue={formData.recovery_phone_country_code || "91"} />
                </div>}
                <Input label="Recovery Phone" value={formData.recovery_phone} onChange={(v) => handleChange('recovery_phone', v)} />
              </div>
            </div>

            <div className="mt-6 pt-6 border-t border-border/50">
              <p className="text-[10px] font-bold text-muted-foreground uppercase mb-4 tracking-tighter">Verification Status</p>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-y-4 gap-x-8">
                <Toggle label="Email Verified" checked={formData.primary_email_verified} onChange={(v) => handleChange('primary_email_verified', v)} />
                <Toggle label="Phone Verified" checked={formData.is_verified} onChange={(v) => handleChange('is_verified', v)} />
                <Toggle label="Recovery Email Verified" checked={formData.recovery_email_verified} onChange={(v) => handleChange('recovery_email_verified', v)} />
                <Toggle label="Recovery Phone Verified" checked={formData.recovery_phone_verified} onChange={(v) => handleChange('recovery_phone_verified', v)} />
              </div>
            </div>
          </CategoryCard>

          {/* Section: System Control */}
          <CategoryCard title="Access & Security" icon={<Shield size={16} />} variant="warning">
            <div className="flex flex-col gap-6">
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 bg-secondary/20 p-4 rounded-3xl border border-border/40">
                {/* <Toggle label="Administrator" checked={formData.is_admin} onChange={(v) => handleChange('is_admin', v)} /> */}
                <Toggle label="Active Status" checked={formData.is_active} onChange={(v) => handleChange('is_active', v)} />
                {/* <Toggle label="Suspended" checked={()=> !formData.is_active} onChange={(v) => handleChange('is_active', v)} /> */}
              </div>

              <div className="space-y-2">
                <label className="text-[11px] font-bold text-muted-foreground uppercase ml-1 flex items-center gap-1">
                  Suspension Reason <Info size={12} />
                </label>
                <textarea
                  className="w-full bg-secondary/30 border border-border/50 rounded-2xl px-4 py-3 text-sm min-h-[80px] outline-none focus:ring-2 focus:ring-amber-500/20"
                  placeholder="State the reason for account suspension..."
                  value={formData.suspension_reason || ""}
                  onChange={(e) => handleChange('suspension_reason', e.target.value)}
                />
              </div>
            </div>
          </CategoryCard>
        </div>

        {/* Footer */}
        <div className="p-6 border-t border-border flex flex-col sm:flex-row gap-3 bg-secondary/5">
          <button onClick={handleOnClose} className="flex-1 px-5 py-4 rounded-2xl font-bold text-sm hover:bg-secondary transition-all order-2 sm:order-1 active:scale-[0.98]">
            Cancel
          </button>
          {user?.user_id  ? <button
            onClick={() => { console.log(formData); onSave(formData) }}
            className="flex-[2] bg-primary text-white px-6 py-4 rounded-2xl font-bold text-sm flex items-center justify-center gap-2 hover:opacity-90 shadow-lg shadow-primary/20 transition-all order-1 sm:order-2 active:scale-[0.98]"
          >
            <Save size={18} /> Update Configuration
          </button> :
            <button
              onClick={() => { console.log(formData); handleCreateUser(formData) }}
              className="flex-[2] bg-primary text-white px-6 py-4 rounded-2xl font-bold text-sm flex items-center justify-center gap-2 hover:opacity-90 shadow-lg shadow-primary/20 transition-all order-1 sm:order-2 active:scale-[0.98]"
            >
              {isSubmitting ? (
                <>
                  <Loader2 size={18} className="animate-spin" />
                    Adding User...
                  </>
              ) : (
                  <>
                    <Save size={18} />
                      Add User
                    </>
                )}
            </button>
          }
        </div>
      </div>
    </div>
  );
}

/* --- Sub-Components --- */

const CategoryCard = ({ title, icon, children, variant = "default" }: any) => (
  <div className="space-y-4">
    <div className="flex items-center gap-2 px-1">
      <span className={cn(
        "p-1.5 rounded-lg",
        variant === "warning" ? "bg-amber-500/10 text-amber-600" : "bg-primary/10 text-primary"
      )}>{icon}</span>
      <h3 className="text-sm font-bold tracking-tight opacity-80 uppercase">{title}</h3>
    </div>
    <div className="bg-background border border-border/60 rounded-[2rem] p-5 sm:p-7 shadow-sm">
      {children}
    </div>
  </div>
);

const Input = ({ label, value, onChange, type = "text", className = "" }: any) => (
  <div className="flex flex-col gap-2">
    <label className="text-[11px] font-bold text-muted-foreground uppercase ml-1">{label}</label>
    <div className="relative w-full">
      {/* {label?.includes("Phone") && <div className="absolute z-1 left-4 top-1/2 -translate-y-1/2 flex items-center gap-2 border-r border-border pr-2 text-muted-foreground">
        <CountrySelector onChange={() => { }} defaultValue={"91"} />
      </div>} */}
      <input
        type={type}
        className={cn(
          "w-full bg-secondary/30 border border-border/50 rounded-2xl px-4 py-3 text-sm outline-none focus:ring-2 focus:ring-primary/20 transition-all placeholder:text-muted-foreground/50",
          className,
          label?.includes("Phone") && "pl-[140px]"
        )}
        value={value || ""}
        onChange={(e) => onChange(e.target.value)}
      />
    </div>
  </div>
);

const Toggle = ({ label, checked, onChange }: any) => (
  <label className="flex items-center justify-between gap-4 cursor-pointer group">
    <span className="text-sm font-semibold text-foreground/80 group-hover:text-foreground transition-colors">{label}</span>
    <div
      onClick={() => onChange(!checked)}
      className={cn(
        "relative w-12 h-6 rounded-full transition-all duration-300 ease-in-out shrink-0",
        checked ? "bg-primary" : "bg-muted-foreground/20"
      )}
    >
      <div className={cn(
        "absolute top-1 left-1 w-4 h-4 bg-white rounded-full transition-all duration-300 shadow-sm",
        checked ? "translate-x-6" : "translate-x-0"
      )} />
    </div>
  </label>
);


// "use client";
// import { useState, useEffect } from "react";
// import { X, Save, Shield, Mail, Phone, Lock, Calendar, User } from "lucide-react";
// import { cn } from "@/lib/utils";

// export default function EditUserModal({ user, isOpen, onClose, onSave }: any) {
//     const [formData, setFormData] = useState<any>(null);

//     useEffect(() => {
//         if (user) setFormData({ ...user });
//     }, [user]);

//     if (!isOpen || !formData) return null;

//     const handleChange = (field: string, value: any) => {
//         setFormData((prev: any) => ({ ...prev, [field]: value }));
//     };

//     return (
//         <div className="fixed inset-0 z-[100] flex items-center justify-center p-4 bg-black/60 backdrop-blur-sm">
//             <div className="bg-background border border-border w-full max-w-2xl max-h-[90vh] overflow-hidden rounded-3xl shadow-2xl flex flex-col">
//                 {/* Header */}
//                 <div className="p-6 border-b border-border flex justify-between items-center bg-secondary/10">
//                     <div>
//                         <h2 className="text-xl font-bold">Edit User Profile</h2>
//                         <p className="text-xs text-muted-foreground uppercase tracking-wider mt-1">ID: {formData.id}</p>
//                     </div>
//                     <button onClick={onClose} className="p-2 hover:bg-secondary rounded-full transition-colors">
//                         <X size={20} />
//                     </button>
//                 </div>

//                 {/* Scrollable Content */}
//                 <div className="p-6 overflow-y-auto space-y-8">

//                     {/* Section: Basic Profile */}
//                     <section className="space-y-4">
//                         <h3 className="text-sm font-semibold flex items-center gap-2 text-primary">
//                             <User size={16} /> Basic Profile
//             </h3>
//                         <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//                             <Input label="First Name" value={formData.first_name} onChange={(v) => handleChange('first_name', v)} />
//                             <Input label="Last Name" value={formData.last_name} onChange={(v) => handleChange('last_name', v)} />
//                             <div className="flex flex-col gap-1.5">
//                                 <label className="text-xs font-medium text-muted-foreground ml-1">Gender</label>
//                                 <select
//                                     className="bg-secondary/50 border-none rounded-xl px-4 py-2.5 text-sm outline-none focus:ring-2 focus:ring-primary/20"
//                                     value={formData.gender || ""}
//                                     onChange={(e) => handleChange('gender', e.target.value)}
//                                 >
//                                     <option value="">Not Specified</option>
//                                     <option value="male">Male</option>
//                                     <option value="female">Female</option>
//                                     <option value="other">Other</option>
//                                 </select>
//                             </div>
//                             <Input label="DOB" type="date" value={formData.dob} onChange={(v) => handleChange('dob', v)} />
//                         </div>
//                     </section>

//                     {/* Section: Contact & Verification */}
//                     <section className="space-y-4">
//                         <h3 className="text-sm font-semibold flex items-center gap-2 text-primary">
//                             <Mail size={16} /> Contact Information
//             </h3>
//                         <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
//                             <Input label="Primary Email" value={formData.primary_email} onChange={(v) => handleChange('primary_email', v)} />
//                             <Input label="Phone Number" value={formData.phone_number} onChange={(v) => handleChange('phone_number', v)} />
//                             <Input label="Recovery Email" value={formData.recovery_email} onChange={(v) => handleChange('recovery_email', v)} />
//                             <Input label="Recovery Phone" value={formData.recovery_phone} onChange={(v) => handleChange('recovery_phone', v)} />
//                         </div>
//                         {/* Verification Toggles */}
//                         <div className="flex flex-wrap gap-3 mt-2">
//                             <Checkbox label="Email Verified" checked={formData.primary_email_verified} onChange={(v) => handleChange('primary_email_verified', v)} />
//                             <Checkbox label="Phone Verified" checked={formData.is_verified} onChange={(v) => handleChange('is_verified', v)} />
//                         </div>
//                         <div className="flex flex-wrap gap-3 mt-2">
//                             <Checkbox label="Recovery Email Verified" checked={formData.primary_email_verified} onChange={(v) => handleChange('primary_email_verified', v)} />
//                             <Checkbox label="Recovery Phone Verified" checked={formData.primary_email_verified} onChange={(v) => handleChange('primary_email_verified', v)} />
//                         </div>
//                     </section>

//                     {/* Section: Admin & Access */}
//                     <section className="p-4 bg-secondary/20 rounded-2xl space-y-4 border border-border">
//                         <h3 className="text-sm font-semibold flex items-center gap-2">
//                             <Shield size={16} /> Permissions & System
//             </h3>
//                         <div className="flex flex-wrap gap-6">
//                             <Checkbox label="Is Administrator" checked={formData.is_admin} onChange={(v) => handleChange('is_admin', v)} />
//                             <Checkbox label="Is Active" checked={formData.is_active} onChange={(v) => handleChange('is_active', v)} />
//                         </div>
//                         <div className="flex flex-wrap gap-6">
//                             <Checkbox label="Is Suspended" checked={formData.is_admin} onChange={(v) => handleChange('is_admin', v)} />
//                         </div>
//                         <div>
//                             <Input label="Suspension reason" value={formData.last_name} onChange={(v) => handleChange('last_name', v)} />
//                         </div>
//                     </section>
//                 </div>

//                 {/* Footer */}
//                 <div className="p-6 border-t border-border flex justify-end gap-3 bg-secondary/10">
//                     <button onClick={onClose} className="px-5 py-2 rounded-xl font-medium hover:bg-secondary transition-colors">Cancel</button>
//                     <button
//                         onClick={() => onSave(formData)}
//                         className="bg-primary text-white px-6 py-2 rounded-xl font-medium flex items-center gap-2 hover:opacity-90 transition-opacity"
//                     >
//                         <Save size={18} /> Update User
//           </button>
//                 </div>
//             </div>
//         </div>
//     );
// }

// // Small helper components
// const Input = ({ label, value, onChange, type = "text", readOnly = false, className = "" }: any) => (
//     <div className="flex flex-col gap-1.5">
//         <label className="text-xs font-medium text-muted-foreground ml-1">{label}</label>
//         <input
//             type={type}
//             readOnly={readOnly}
//             className={cn(
//                 "bg-secondary/50 border-none rounded-xl px-4 py-2.5 text-sm outline-none focus:ring-2 focus:ring-primary/20 transition-all",
//                 className
//             )}
//             value={value || ""}
//             onChange={(e) => onChange(e.target.value)}
//         />
//     </div>
// );

// const Checkbox = ({ label, checked, onChange }: any) => (
//     <label className="flex items-center gap-2 cursor-pointer group">
//         <div className={cn(
//             "w-5 h-5 rounded border flex items-center justify-center transition-colors",
//             checked ? "bg-primary border-primary" : "border-border bg-background group-hover:border-primary/50"
//         )}>
//             <input type="checkbox" className="hidden" checked={checked} onChange={(e) => onChange(e.target.checked)} />
//             {checked && <Save size={12} className="text-white" />}
//         </div>
//         <span className="text-sm font-medium">{label}</span>
//     </label>
// );