"use client";

import Link from "next/link";
import { usePathname } from "next/navigation";
import {
  Home, User, Share2, X, ChevronRight, ChevronLeft, Monitor,
  LayoutDashboard, Users, Grid, BarChart3, Landmark, ShieldCheck, FileClock
} from "lucide-react";
import { cn } from "@/lib/utils";
import { useState } from "react";

const menuItems = [
  { icon: Home, label: "Home", href: "/dashboard" },
  { icon: User, label: "Personal info", href: "/settings" },
  { icon: Share2, label: "Connected Apps", href: "/connections" },
  { icon: Monitor, label: "Your Devices", href: "/devices" }
];

// Added Admin specific items
const adminItems = [
  { icon: LayoutDashboard, label: "Admin Dashboard", href: "/admin/dashboard" },
  { icon: Users, label: "Users", href: "/admin/users" },
  { icon: Grid, label: "Apps", href: "/admin/apps" },
  { icon: FileClock, label: "Audit Logs", href: "/admin/logs" },
  { icon: BarChart3, label: "Analytics", href: "/admin/analytics" },
  { icon: Landmark, label: "Finance", href: "/admin/finance" },
];

interface SidebarProps {
  isOpen: boolean;
  onClose: () => void;
  isAdmin?: boolean; // New prop to handle role
}

export const Sidebar = ({ isOpen, onClose, isAdmin = true }: SidebarProps) => {
  const [isCollapsed, setIsCollapsed] = useState(false);
  const pathname = usePathname();

  // Helper to render navigation links
  const renderLinks = (items: typeof menuItems) => (
    items.map((item) => {
      const isActive = pathname === item.href;
      return (
        <Link
          key={item.href}
          href={item.href}
          onClick={() => { if (isOpen) onClose(); }}
          className={cn(
            "flex items-center transition-all duration-200 rounded-xl text-sm font-medium group",
            isCollapsed ? "md:justify-center md:px-0" : "px-4 py-3 space-x-4",
            "py-4 px-4 space-x-4 md:py-3",
            isActive
              ? "bg-primary text-primary-foreground shadow-lg shadow-primary/20"
              : "hover:bg-secondary text-muted-foreground hover:text-foreground"
          )}
        >
          <item.icon size={22} className={cn("shrink-0", !isActive && "group-hover:scale-110 transition-transform")} />
          {!isCollapsed && (
            <span className="whitespace-nowrap opacity-100 transition-all">
              {item.label}
            </span>
          )}
        </Link>
      );
    })
  );

  return (
    <>
      {/* 1. Mobile Overlay */}
      {isOpen && (
        <div
          className="fixed inset-0 bg-background/80 z-[60] md:hidden backdrop-blur-md transition-opacity"
          onClick={onClose}
        />
      )}

      <aside className={cn(
        "fixed left-0 top-0 h-full bg-background border-r border-border transition-all duration-300 ease-in-out z-[70]",
        "md:sticky md:top-16 md:z-40",
        "h-[100dvh] md:h-[calc(100dvh-72px)]",
        "w-[85%] sm:w-80 md:w-72", 
        isCollapsed ? "md:w-20" : "md:w-72",
        isOpen ? "translate-x-0 shadow-2xl" : "-translate-x-full md:translate-x-0"
      )}>
        
        <button
          onClick={() => setIsCollapsed(!isCollapsed)}
          className="absolute -right-3 top-4 bg-background border border-border rounded-full p-1 hover:bg-accent text-muted-foreground shadow-sm hidden md:block"
        >
          {isCollapsed ? <ChevronRight size={16} /> : <ChevronLeft size={16} />}
        </button>

        {/* Mobile Header */}
        <div className="flex items-center justify-between p-6 md:hidden border-b border-border">
          <div className="flex items-center gap-2">
            <div className="w-8 h-8 rounded-lg bg-primary flex items-center justify-center text-primary-foreground font-bold">A</div>
            <span className="font-bold text-foreground">Menu</span>
          </div>
          <button onClick={onClose} className="p-2 hover:bg-secondary rounded-xl text-muted-foreground">
            <X size={20} />
          </button>
        </div>

        {/* Navigation Content */}
        <div className="h-full overflow-y-auto custom-scrollbar pb-24">
          <nav className="space-y-1 p-4 mt-2">
            {renderLinks(menuItems)}

            {/* Admin Section */}
            {isAdmin && false && (
              <div className="mt-8 pt-4 border-t border-border/50">
                {!isCollapsed ? (
                  <p className="px-4 mb-2 text-[10px] font-semibold uppercase tracking-wider text-muted-foreground/60 flex items-center gap-2">
                    <ShieldCheck size={12} />
                    Administration
                  </p>
                ) : (
                  <div className="flex justify-center mb-2">
                    <div className="h-px w-8 bg-border" />
                  </div>
                )}
                <div className="space-y-1">
                   {renderLinks(adminItems)}
                </div>
              </div>
            )}
          </nav>
        </div>

        {/* Mobile Footer */}
        <div className="absolute bottom-6 left-0 w-full px-4 pointer-events-none">
           <p className="text-center text-[10px] uppercase tracking-widest text-muted-foreground/40">
             Version 1.0.4
           </p>
        </div>
      </aside>
    </>
  );
};

// "use client";

// import Link from "next/link";
// import { usePathname } from "next/navigation";
// import {
//   Home, User, Share2, X, ChevronRight, ChevronLeft, Monitor
// } from "lucide-react";
// import { cn } from "@/lib/utils";
// import { useState } from "react";

// const menuItems = [
//   { icon: Home, label: "Home", href: "/dashboard" },
//   { icon: User, label: "Personal info", href: "/settings" },
//   { icon: Share2, label: "Connected Apps", href: "/connections" },
//   { icon: Monitor, label: "Your Devices", href: "/devices" }

// ];

// interface SidebarProps {
//   isOpen: boolean;
//   onClose: () => void;
// }

// export const Sidebar = ({ isOpen, onClose }: SidebarProps) => {
//   const [isCollapsed, setIsCollapsed] = useState(false);
//   const pathname = usePathname();

//   return (
//     <>
//       {/* 1. Mobile Overlay (Blur & Darken) */}
//       {isOpen && (
//         <div
//           className="fixed inset-0 bg-background/80 z-[60] md:hidden backdrop-blur-md transition-opacity"
//           onClick={onClose}
//         />
//       )}

//       <aside className={cn(
//         // Basic Styles
//         "fixed left-0 top-0 h-full bg-background border-r border-border transition-all duration-300 ease-in-out z-[70]",
        
//         // Desktop positioning (After Navbar)
//         "md:sticky md:top-16 md:z-40",
        
//         "h-[100dvh] md:h-[calc(100dvh-72px)]",
        
//         // Width Logic: Mobile is almost full-width, Desktop respects collapse state
//         "w-[85%] sm:w-80 md:w-72", 
//         isCollapsed ? "md:w-20" : "md:w-72",
        
//         // Show/Hide Logic
//         isOpen ? "translate-x-0 shadow-2xl" : "-translate-x-full md:translate-x-0"
//       )}
//         // style={{
//         //   // On mobile, take full height. On desktop, subtract navbar height.
//         //   height: 'calc(100dvh - 72px)', // Default to dynamic viewport height
//         // }}
       
//       >
//         {/* Collapse Toggle - Hidden on mobile, only desktop */}
//         <button
//           onClick={() => setIsCollapsed(!isCollapsed)}
//           className="absolute -right-3 top-4 bg-background border border-border rounded-full p-1 hover:bg-accent text-muted-foreground shadow-sm hidden md:block"
//         >
//           {isCollapsed ? <ChevronRight size={16} /> : <ChevronLeft size={16} />}
//         </button>

//         {/* 2. Mobile Drawer Header (Cleaner Brand/Label) */}
//         <div className="flex items-center justify-between p-6 md:hidden border-b border-border">
//           <div className="flex items-center gap-2">
//             <div className="w-8 h-8 rounded-lg bg-primary flex items-center justify-center text-primary-foreground font-bold">A</div>
//             <span className="font-bold text-foreground">Menu</span>
//           </div>
//           <button 
//             onClick={onClose} 
//             className="p-2 hover:bg-secondary rounded-xl text-muted-foreground"
//           >
//             <X size={20} />
//           </button>
//         </div>

//         {/* Navigation Links */}
//         <nav className="space-y-2 p-4 mt-2">
//           {menuItems.map((item) => {
//             const isActive = pathname === item.href;

//             return (
//               <Link
//                 key={item.href}
//                 href={item.href}
//                 onClick={() => {
//                   if (isOpen) onClose();
//                 }}
//                 className={cn(
//                   "flex items-center transition-all duration-200 rounded-xl text-sm font-medium",
//                   // Desktop Collapse spacing
//                   isCollapsed ? "md:justify-center md:px-0" : "px-4 py-3.5 space-x-4",
//                   // Mobile spacing (always wide)
//                   "py-4 px-4 space-x-4 md:py-3",
//                   isActive
//                     ? "bg-primary text-primary-foreground shadow-lg shadow-primary/20"
//                     : "hover:bg-secondary text-muted-foreground hover:text-foreground"
//                 )}
//               >
//                 <item.icon size={22} className="shrink-0" />
//                 {!isCollapsed && <span className={cn(
//                   "whitespace-nowrap transition-all",
//                   isCollapsed ? "md:opacity-0 md:w-0 overflow-hidden" : "opacity-100"
//                 )}>
//                   {item.label}
//                 </span>}
//               </Link>
//             );
//           })}
//         </nav>

//         {/* 3. Mobile Footer (Optional: Logout or version info) */}
//         <div className="absolute bottom-8 left-0 w-full px-4">
//            <p className="text-center text-[10px] uppercase tracking-widest text-muted-foreground opacity-50">
//              Version 1.0.4
//            </p>
//         </div>
//       </aside>
//     </>
//   );
// };

