"use client";
import { useState, useRef, useEffect } from "react";
import {
    Sun, Moon, Grid, X, Camera, Plus, LogOut,
    ChevronDown, ChevronUp, Home, Check, ChevronRight, Loader2, Trash2, UserMinus, User
} from "lucide-react";
import { cn } from "@/lib/utils";
import Link from "next/link";
import { useAuth } from "@/context/AuthContext"
import { useRouter } from 'next/navigation';
import { authService } from "@/lib/auth"
interface Account {
    email: string;
    name: string;
    image?: string;
    isLoggedIn?: boolean;
    user_id?: string;
}

interface ApiAccount {
    user_id: string;
    identifier: string;
    active: boolean;
}

interface AccountSwitcherProps {
    primaryAccount: Account;
    theme: string;
    toggleTheme: () => void;
}

export default function AccountSwitcher({ primaryAccount, theme, toggleTheme }: AccountSwitcherProps) {
    const [isOpen, setIsOpen] = useState(false);
    const [showMore, setShowMore] = useState(true);
    const [mergedAccounts, setMergedAccounts] = useState<Account[]>([]);
    const [fetchingAccounts, setFetchingAccounts] = useState(false);

    // Tracks which account is "active/clicked" on mobile to keep the delete button visible
    const [selectedEmail, setSelectedEmail] = useState<string | null>(null);

    const [switchingId, setSwitchingId] = useState<string | null>(null);
    const [removingEmail, setRemovingEmail] = useState<string | null>(null);

    const menuRef = useRef<HTMLDivElement>(null);
    const { user, loading, logout } = useAuth();
    const router = useRouter();

    const handleLogOut = async () => {
        // logout();
        const response = await authService.logoutAccount()
        if (response.success) {
            window.location.reload();
        }
        setIsOpen(false);
    };

    const validateDevice = () => {
        const deviceId = localStorage.getItem("device_id");
        if (!deviceId) {
            router.push("/login");
            return null;
        }
        return deviceId;
    };

    useEffect(() => {
        validateDevice();
    }, []);

    const handleSwitchAccount = async (userId: string) => {
        const device_id = validateDevice()
        setSwitchingId(userId);
        try {
            const response = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/switch-account/`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-Device-Id': device_id || "",
                },
                credentials: "include",
                body: JSON.stringify({ user_id: userId })
            });
            if (response.ok) {
                window.location.reload();
            }
        } catch (error) {
            console.error("Failed to switch account", error);
        } finally {
            setSwitchingId(null);
        }
    };

    // const handleRemoveAccount = async (e: React.MouseEvent, acc: Account) => {
    //     e.stopPropagation(); 
    //     setRemovingEmail(acc.email);

    //     try {
    //         if (acc.isLoggedIn) {
    //             const device_id = validateDevice();
    //             await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/account/delete/`, {
    //                 method: 'POST',
    //                 headers: {
    //                     'Content-Type': 'application/json',
    //                     'X-Device-Id': device_id || "",
    //                 },
    //                 credentials:"include",
    //                 body: JSON.stringify({ user_id: acc.user_id })
    //             });

    //         }

    //         const saved = localStorage.getItem("saved_accounts");
    //         if (saved) {
    //             const localAccounts: Account[] = JSON.parse(saved);
    //             const updated = localAccounts.filter(a => a.email !== acc.email);
    //             localStorage.setItem("saved_accounts", JSON.stringify(updated));
    //         }

    //         setMergedAccounts(prev => prev.filter(a => a.email !== acc.email));
    //         setSelectedEmail(null);
    //     } catch (error) {
    //         console.error("Error removing account", error);
    //     } finally {
    //         setRemovingEmail(null);
    //     }
    // };

    const [reloadAccounts, setReloadAccounts] = useState(false)
    const handleRemoveAccount = async (e: React.MouseEvent, acc: Account) => {
        e.stopPropagation();
        setRemovingEmail(acc.email);

        try {
            if (acc.isLoggedIn) {
                const device_id = validateDevice();

                // 1. Remove from database/backend
                const res = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/account/delete/`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Device-Id': device_id || "",
                    },
                    credentials: "include",
                    body: JSON.stringify({ user_id: acc.user_id })
                });
                const data = await res.json()
                // 2. Sync logic: Ensure it exists in localStorage even after DB removal
                // const saved = localStorage.getItem("saved_accounts");
                // let localAccounts: Account[] = saved ? JSON.parse(saved) : [];

                // const exists = localAccounts.some(a => a.email === acc.email);

                // if (!exists) {
                //     // Add to local storage since it's being removed from the "active" DB session
                //     const accountToSave = { ...acc, isLoggedIn: false }; // Mark as logged out locally
                //     localAccounts.push(accountToSave);
                //     localStorage.setItem("saved_accounts", JSON.stringify(localAccounts));
                // }
                // setMergedAccounts(prev => prev);
                setReloadAccounts(prev => !prev)
                return
            }

            // 3. Update Local Storage (Remove if the intent is to fully delete)
            // Note: If you want to keep it in Local Storage but remove it from the UI list, 
            // you might need to adjust the filtering logic below.
            const saved = localStorage.getItem("saved_accounts");
            if (saved) {
                const localAccounts: Account[] = JSON.parse(saved);
                const updated = localAccounts.filter(a => a.email !== acc.email);
                localStorage.setItem("saved_accounts", JSON.stringify(updated));
            }

            // 4. Update UI State
            setMergedAccounts(prev => prev.filter(a => a.email !== acc.email));
            setSelectedEmail(null);

        } catch (error) {
            console.error("Error removing account", error);
        } finally {
            setRemovingEmail(null);
        }
    };
    useEffect(() => {
        const syncAccounts = async () => {
            const device_id = validateDevice()
            setFetchingAccounts(true);
            try {
                const res = await fetch(`${process.env.NEXT_PUBLIC_API_BASE_URL}/api/auth/accounts/`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Device-Id': device_id || "",
                    },
                });
                const apiAccounts: ApiAccount[] = await res.json();

                const saved = localStorage.getItem("saved_accounts");
                let localAccounts: Account[] = saved ? JSON.parse(saved) : [];

                const otherApiAccounts = apiAccounts
                    .filter(api => !api.active)
                    .map(api => ({
                        email: api.identifier,
                        name: api.identifier.split('@')[0],
                        user_id: api.user_id,
                        isLoggedIn: true
                    }));

                const filteredLocal = localAccounts.filter(local =>
                    local.email !== user?.primary_email &&
                    !apiAccounts.some(api => api.identifier === local.email)
                ).map(local => ({ ...local, isLoggedIn: false }));

                setMergedAccounts([...otherApiAccounts, ...filteredLocal]);
            } catch (e) {
                console.error("Error syncing accounts", e);
            } finally {
                setFetchingAccounts(false);
            }
        };

        if (isOpen && user) {
            syncAccounts();
        }
    }, [isOpen, user, reloadAccounts]);

    useEffect(() => {
        const handler = (e: MouseEvent) => {
            if (menuRef.current && !menuRef.current.contains(e.target as Node)) {
                setIsOpen(false);
                setSelectedEmail(null); // Reset selection when closing
            }
        };
        document.addEventListener("mousedown", handler);
        return () => document.removeEventListener("mousedown", handler);
    }, []);

    const getInitial = (name: string) => name?.charAt(0).toUpperCase() || "?";

    return (
        <div className="flex items-center gap-1 relative" ref={menuRef}>
            {loading ? (
                <div className="ml-2 w-8 h-8 rounded-full flex items-center justify-center">
                    <Loader2 className="h-5 w-5 animate-spin text-sidebar-primary" />
                </div>
            ) : (
                    <div
                        onClick={() => setIsOpen(!isOpen)}
                        className="ml-2 w-8 h-8 rounded-full bg-sidebar-primary flex items-center justify-center text-sm text-white font-medium cursor-pointer border border-border shadow-sm hover:ring-4 hover:ring-sidebar-primary/20 transition-all overflow-hidden"
                    >
                        {!user && <User size={18}/>}
                        {user?.profile_picture ? (
                            <img src={user.profile_picture} alt="Profile" className="w-full h-full object-cover" />
                        ) : (
                                user?.first_name?.charAt(0).toUpperCase()
                            )}
                    </div>
                )}

            {isOpen && user && (
                <div className="absolute overflow-y-auto max-h-[80vh] top-12 right-0 w-[350px] bg-[#f0f4f9] dark:bg-[#1e1f20] rounded-[28px] shadow-2xl z-[100] border border-border/50 p-2 animate-in fade-in zoom-in-95 duration-200">

                    <div className="bg-card rounded-[24px] p-4 text-center relative shadow-sm">
                        <button onClick={() => setIsOpen(false)} className="absolute right-4 top-4 text-muted-foreground hover:bg-accent p-1.5 rounded-full transition-colors">
                            <X size={18} />
                        </button>
                        <p className="text-sm font-medium mb-4 text-foreground/80">{user?.primary_email}</p>
                        <div className="relative w-20 h-20 mx-auto group">
                            <div className="w-full h-full rounded-full bg-green-700 flex items-center justify-center text-3xl text-white overflow-hidden">
                                {user?.profile_picture ? (
                                    <img src={user.profile_picture} alt="Profile" className="w-full h-full object-cover" />
                                ) : (
                                        user?.first_name?.charAt(0).toUpperCase()
                                    )}
                            </div>
                            <Link href="/settings" className="absolute bottom-0 right-0 p-1.5 bg-card border border-border rounded-full shadow-md hover:bg-accent">
                                <Camera size={14} />
                            </Link>
                        </div>
                        <h2 className="text-xl font-normal mt-3 text-foreground">Hi, {user?.first_name}!</h2>
                        <Link href="/settings">
                            <button className="mt-4 px-6 py-2 border border-border rounded-full text-sm font-semibold hover:bg-sidebar-primary/5 transition-colors text-sidebar-primary">
                                Manage your Account
                            </button>
                        </Link>
                    </div>

                    <div className="mt-2 bg-card rounded-[24px] overflow-hidden shadow-sm">
                        <button
                            onClick={() => setShowMore(!showMore)}
                            className="w-full flex items-center justify-between p-4 hover:bg-accent/50 transition-colors"
                        >
                            <span className="text-sm font-medium">Other accounts</span>
                            {showMore ? <ChevronUp size={18} /> : <ChevronDown size={18} />}
                        </button>

                        {showMore && (
                            <div className="pb-2">
                                {fetchingAccounts ? (
                                    <div className="p-4 space-y-4">
                                        {[1, 2].map((i) => (
                                            <div key={i} className="flex items-center gap-3 animate-pulse">
                                                <div className="w-9 h-9 bg-muted rounded-full" />
                                                <div className="flex-1 space-y-2">
                                                    <div className="h-3 bg-muted rounded w-1/2" />
                                                    <div className="h-2 bg-muted rounded w-3/4" />
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                        mergedAccounts.map((acc, i) => (
                                            <div
                                                key={acc.email}
                                                onClick={() => {
                                                    if (switchingId || removingEmail) return;

                                                    // If it's already selected, switch/login. 
                                                    // If not, first tap selects it (shows delete btn)
                                                    if (selectedEmail === acc.email) {
                                                        acc.isLoggedIn ? handleSwitchAccount(acc.user_id!) : router.push(`/login?email=${acc.email}`)
                                                    } else {
                                                        setSelectedEmail(acc.email);
                                                    }
                                                }}
                                                className={cn(
                                                    "flex items-center justify-between px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40 group relative transition-colors",
                                                    selectedEmail === acc.email && "bg-accent/60",
                                                    (switchingId === acc.user_id) && "opacity-70 pointer-events-none"
                                                )}
                                            >
                                                <div className="flex items-center gap-3">
                                                    <div className="w-9 h-9 rounded-full bg-muted flex items-center justify-center text-muted-foreground text-sm font-bold overflow-hidden">
                                                        {switchingId === acc.user_id ? (
                                                            <Loader2 className="h-4 w-4 animate-spin" />
                                                        ) : (
                                                                acc.image ? <img src={acc.image} alt="" /> : getInitial(acc.name)
                                                            )}
                                                    </div>
                                                    <div className="text-left">
                                                        <p className="text-sm font-semibold leading-none text-foreground">{acc.name}</p>
                                                        <p className="text-xs text-muted-foreground mt-1 break-all">{acc.email}</p>
                                                    </div>
                                                </div>

                                                <div className="flex items-center gap-2">
                                                    {!acc.isLoggedIn && <span className="text-[10px] bg-muted px-1.5 py-0.5 rounded text-muted-foreground">Signed out</span>}

                                                    <button
                                                        onClick={(e) => handleRemoveAccount(e, acc)}
                                                        className={cn(
                                                            "p-1.5 rounded-full hover:bg-destructive/10 text-muted-foreground hover:text-red-600 transition-all",
                                                            // Visibile if hovered (Desktop) OR if selected (Mobile)
                                                            "opacity-0 group-hover:opacity-100",
                                                            selectedEmail === acc.email && "opacity-100 bg-destructive/5 text-red-600 scale-110",
                                                            removingEmail === acc.email && "opacity-100"
                                                        )}
                                                        title="Remove account"
                                                    >
                                                        {removingEmail === acc.email ? (
                                                            <Loader2 size={14} className="animate-spin" />
                                                        ) : (
                                                                <UserMinus size={14} />
                                                            )}
                                                    </button>

                                                    <ChevronRight size={16} className={cn(
                                                        "text-muted-foreground transition-opacity",
                                                        selectedEmail === acc.email ? "opacity-100" : "opacity-0 group-hover:opacity-100"
                                                    )} />
                                                </div>
                                            </div>
                                        ))
                                    )}

                                <Link href="/login" className="flex items-center gap-3 px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40">
                                    <div className="w-9 h-9 flex items-center justify-center">
                                        <Plus size={20} className="text-muted-foreground" />
                                    </div>
                                    <span className="text-sm font-medium">Add another account</span>
                                </Link>

                                <button onClick={handleLogOut} className="w-full flex items-center gap-3 px-4 py-3 hover:bg-accent/40 cursor-pointer border-t border-border/40">
                                    <div className="w-9 h-9 flex items-center justify-center">
                                        <LogOut size={20} className="text-muted-foreground" />
                                    </div>
                                    <span className="text-sm font-medium">Sign out</span>
                                </button>
                            </div>
                        )}
                    </div>

                    <div className="flex justify-center gap-4 py-4 text-[10px] text-muted-foreground font-medium uppercase tracking-tighter">
                        <span className="hover:underline cursor-pointer"><Link href="/help">Help</Link></span>
                        <span>•</span>
                        <span className="hover:underline cursor-pointer">Privacy</span>
                        <span>•</span>
                        <span className="hover:underline cursor-pointer">Terms</span>
                    </div>
                </div>
            )}
        </div>
    );
}
