"use client";
import { useState, useRef, useEffect } from "react";
import { Search, X, ChevronRight, Loader2, LucideIcon } from "lucide-react";
import { cn } from "@/lib/utils";
import Link from "next/link";
import * as Icons from "lucide-react";

interface SearchItem {
    id: string;
    title: string;
    path: string;
    keywords: string[];
    icon: keyof typeof Icons; // Dynamic icon lookup
}

interface GlobalSearchProps {
    data: SearchItem[];
    placeholder?: string;
    onSelect?: (item: SearchItem) => void;
}

export default function GlobalSearch({ data, placeholder = "Search...", onSelect }: GlobalSearchProps) {
    const [query, setQuery] = useState("");
    const [results, setResults] = useState<SearchItem[]>([]);
    const [isOpen, setIsOpen] = useState(false);
    const [isSearching, setIsSearching] = useState(false);
    const containerRef = useRef<HTMLDivElement>(null);

    // --- Debounce Logic ---
    useEffect(() => {
        if (!query.trim()) {
            setResults([]);
            setIsOpen(false);
            setIsSearching(false);
            return;
        }

        setIsSearching(true);
        const timer = setTimeout(() => {
            const filtered = data.filter(item =>
                item.title.toLowerCase().includes(query.toLowerCase()) ||
                item.keywords.some(k => k.toLowerCase().includes(query.toLowerCase()))
            );

            setResults(filtered);
            setIsSearching(false);
            setIsOpen(true);
        }, 350);

        return () => clearTimeout(timer);
    }, [query, data]);

    // Click Outside Handler
    useEffect(() => {
        const handleClick = (e: MouseEvent) => {
            if (containerRef.current && !containerRef.current.contains(e.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClick);
        return () => document.removeEventListener("mousedown", handleClick);
    }, []);



    return (
        <div className="flex-1 max-w-2xl mx-auto w-full relative" ref={containerRef}>
            <div className="relative group">
                <div className="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                    {isSearching ? (
                        <Loader2 className="h-4 w-4 text-sidebar-primary animate-spin" />
                    ) : (
                            <Search className={cn("h-4 w-4 transition-colors", isOpen ? "text-sidebar-primary" : "text-muted-foreground")} />
                        )}
                </div>

                <input
                    type="text"
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    onFocus={() => query.length > 0 && setIsOpen(true)}
                    className="w-full pl-12 pr-4 py-3.5 rounded-full border border-border bg-background focus:outline-none focus:ring-2 focus:ring-ring transition-all"
                    //   className="block w-full pl-11 pr-12 py-2.5 bg-accent/40 border-none rounded-full focus:bg-background focus:ring-2 focus:ring-sidebar-primary/20 transition-all placeholder:text-muted-foreground text-sm outline-none"
                    placeholder={placeholder}
                />

                {query && (
                    <button
                        onClick={() => setQuery("")}
                        className="absolute inset-y-0 right-3 flex items-center px-2 text-muted-foreground hover:text-foreground"
                    >
                        <X size={14} />
                    </button>
                )}
            </div>

            {/* RESULTS OVERLAY */}
            {isOpen && (
                <div className="absolute top-full left-0 right-0 mt-2 bg-card border border-border rounded-2xl shadow-2xl overflow-hidden z-[100] animate-in fade-in slide-in-from-top-2">
                    <div className="max-h-[400px] overflow-y-auto p-2 scrollbar-hide">
                        {results.length > 0 && <div className="px-3 py-2 text-[10px] font-bold text-muted-foreground uppercase tracking-widest flex justify-between items-center">
                            <span>Results</span>
                            <span className="bg-muted px-1.5 py-0.5 rounded text-[8px]">{results.length} items</span>
                        </div>}

                        {results.length > 0 ? (
                            results.map((item) => {
                                // @ts-ignore - Dynamic Lucide Icon Resolution
                                const IconNode = Icons[item.icon] as LucideIcon;

                                return (
                                    <Link
                                        key={item.id}
                                        href={item.path}
                                        onClick={() => {
                                            setIsOpen(false);
                                            if (onSelect) onSelect(item);
                                        }}
                                        className="flex items-center justify-between p-3 hover:bg-sidebar-primary/5 rounded-xl transition-all group mb-1 last:mb-0"
                                    >
                                        <div className="flex items-center gap-4">
                                            <div className="w-9 h-9 rounded-lg bg-sidebar-primary/10 flex items-center justify-center text-sidebar-primary group-hover:bg-sidebar-primary group-hover:text-white transition-all">
                                                {IconNode ? <IconNode size={18} /> : <Search size={18} />}
                                            </div>
                                            <div>
                                                <p className="text-sm font-medium text-foreground">{item.title}</p>
                                                <p className="text-[11px] text-muted-foreground font-light">
                                                    {item.keywords.slice(0, 3).join(" • ")}
                                                </p>
                                            </div>
                                        </div>
                                        <ChevronRight size={14} className="text-muted-foreground opacity-0 group-hover:opacity-100 group-hover:translate-x-1 transition-all" />
                                    </Link>
                                );
                            })
                        ) : (
                                !isSearching && (
                                    <div className="py-10 text-center">
                                        <p className="text-sm text-muted-foreground">No matches found for "{query}"</p>
                                    </div>
                                )
                            )}
                    </div>

                    <div className="bg-muted/30 p-3 border-t border-border flex justify-between items-center">
                        <span className="text-[10px] text-muted-foreground font-medium">Search Powered by Indzs AI</span>
                        <Link href="/help" className="text-[10px] font-bold text-sidebar-primary hover:underline">Support Center</Link>
                    </div>

                </div>
            )}
        </div>
    );
}