"use client";

import React, { createContext, useContext, useEffect, useState, ReactNode } from "react";
import { authService, User } from "@/lib/auth";
import { useRouter, usePathname } from "next/navigation";

interface AuthContextType {
    user: User | null;
    loading: boolean;
    refreshUser: () => Promise<void>;
    logout: () => Promise<void>;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export const AuthProvider = ({ children }: { children: ReactNode }) => {
    const [user, setUser] = useState<User | null>(null);
    const [loading, setLoading] = useState(true);
    const router = useRouter();
    const pathname = usePathname();

    const refreshUser = async () => {
        try {
            const response = await authService.getUserInfo();
            console.log(response)
            if (response.authenticated && response.user) {
                setUser(response.user);
            } else {
                setUser(null);
            }
        } catch (error) {
            console.log(error)
            setUser(null);
        } finally {
            setLoading(false);
        }
    };

    const logout = async () => {
        setLoading(true);
        await authService.logout();
        setUser(null);
        setLoading(false);
        router.push("/login");
    };

    // Initial load
    useEffect(() => {
        refreshUser();
    }, []);

    return (
        <AuthContext.Provider value={{ user, loading, refreshUser, logout }}>
            {children}
        </AuthContext.Provider>
    );
};

export const useAuth = () => {
    const context = useContext(AuthContext);
    if (context === undefined) {
        throw new Error("useAuth must be used within an AuthProvider");
    }
    return context;
};

// "use client"

// import {
//     createContext,
//     useContext,
//     useEffect,
//     useState,
//     ReactNode,
//     Dispatch,
//     SetStateAction,

// } from "react"
// import { useRouter } from 'next/navigation';

// export interface User {
//     // Identity & Core Fields
//     id: string;
//     first_name: string; // Matches Django API
//     last_name : string;
//     phone_number: string;
//     recovery_phone?: string;
//     recovery_phone_verified: boolean;
//     dob: string; 
//     primary_email: string;
//     primary_email_verified: boolean;
//     recovery_email?: string;
//     recovery_email_verified: boolean;


//     // Profile Details
//     firstName: string;   // Local display name
//     lastName: string;
//     gender: "Male" | "Female" | "Other" | "Prefer not to say";
//     birthday: string;    // ISO date string (e.g., "2002-08-10")

//     // Phone Information
//     primaryPhone: string;
//     primaryPhoneVerified: boolean;
//     recoveryPhone?: string;
//     recoveryPhoneVerified: boolean;

//     // Email Information
//     primaryEmail: string;
//     primaryEmailVerified: boolean;
//     recoveryEmail?: string;
//     recoveryEmailVerified: boolean;
// }

// type SetUserType = Dispatch<SetStateAction<User | null>>;

// interface AuthContextType {
//     user: User | null
//     loading: boolean
//     isAuthenticated: boolean
//     refreshUser: () => Promise<void>
//     logout: () => Promise<void>
//     setUser: SetUserType
// }

// const AuthContext = createContext<AuthContextType | null>(null)

// export function AuthProvider({ children }: { children: ReactNode }) {
//     const [user, setUser] = useState<User | null>(null)
//     const [loading, setLoading] = useState(true)
//     const baseUrl = process.env.NEXT_PUBLIC_API_BASE_URL
//     const router = useRouter();
//     // Move this check inside useEffect or a more appropriate place if needed, 
//     // but usually, it's fine as long as the env is set.
//     if (!baseUrl) {
//         console.error("Base URL must be configured")
//     }

//     const refreshUser = async () => {
//         setLoading(true)
//         try {
//             // 1. Try to get user info
//             const res = await fetch(`${baseUrl}/api/auth/me/`, {
//                 method: "GET",
//                 credentials: "include",
//             })

//             // 2. If unauthorized, try to refresh the token
//             if (res.status === 401) {
//                 const refreshRes = await fetch(`${baseUrl}/api/auth/refresh/`, {
//                     method: "POST",
//                     headers: { "Content-Type": "application/json" },
//                     credentials: "include",
//                 });

//                 if (refreshRes.ok) {
//                     // 3. Refresh succeeded! Now RETRY the /me call
//                     return await refreshUser();
//                 } else {
//                     // Refresh failed (expired refresh token)
//                     setUser(null);
//                     return;
//                 }
//             }

//             if (res.ok) {
//                 const data = await res.json();
//                 // Check if your API returns { user: ... } or just the user object
//                 setUser(data.user || data);
//             } else {
//                 setUser(null);
//             }

//         } catch (err) {
//             console.error("Auth refresh error:", err);
//             setUser(null);
//         } finally {
//             setLoading(false);
//         }
//     }

//     const logout = async () => {
//         try {
//             await fetch(`${baseUrl}/api/auth/logout/`, {
//                 method: "POST",
//                 credentials: "include",
//             })
//         } finally {
//             // setUser(null)
//             router.push('/login')
//         }
//     }

//     useEffect(() => {
//         refreshUser()
//     }, [])

//     return (
//         <AuthContext.Provider
//             value={{
//                 user,
//                 setUser,
//                 loading,
//                 isAuthenticated: !!user,
//                 refreshUser,
//                 logout,
//             }}
//         >
//             {children}
//         </AuthContext.Provider>
//     )
// }

// export function useAuth() {
//     const ctx = useContext(AuthContext)
//     if (!ctx) {
//         throw new Error("useAuth must be used inside AuthProvider")
//     }
//     return ctx
// }