import { useCallback, useRef, useState } from "react";
import { useDebounce } from "@/hooks/useDebounce";

const API_BASE = process.env.NEXT_PUBLIC_API_BASE;

export function useAdminUserSearch(resultLimit = 10) {
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [hasNext, setHasNext] = useState(false);
  const [nextCursor, setNextCursor] = useState(null);
  const [searchQuery, setSearchQuery] = useState("");
  const [activeFilter, setActiveFilter] = useState("latest");

  const debouncedQuery = useDebounce(searchQuery, 400);
  const abortRef = useRef(null);

  const fetchSearchUsers = useCallback(
    async (reset = false) => {
      if (loading) return;
      if (!hasNext && !reset) return;
      if (!debouncedQuery) return;

      // Abort previous request
      if (abortRef.current) {
        abortRef.current.abort();
      }

      const controller = new AbortController();
      abortRef.current = controller;

      setLoading(true);

      try {
        const body = {
          query: debouncedQuery,
          limit: resultLimit,
          filter: activeFilter, // future-safe
          ...(reset ? {} : nextCursor && { cursor: nextCursor })
        };

        const res = await fetch(
          `${API_BASE}/api/admin/users/search/`,
          {
            method: "POST",
            credentials: "include",
            headers: {
              "Content-Type": "application/json"
            },
            body: JSON.stringify(body),
            signal: controller.signal
          }
        );

        if (!res.ok) throw new Error("Search failed");

        const data = await res.json();

        setUsers(prev =>
          reset ? data.results : [...prev, ...data.results]
        );
        setNextCursor(data.pagination?.next_cursor || null);
        setHasNext(data.pagination?.has_next ?? false);
      } catch (err) {
        if (err.name !== "AbortError") {
          console.error(err);
        }
      } finally {
        setLoading(false);
      }
    },
    [
      debouncedQuery,
      nextCursor,
      hasNext,
      loading,
      resultLimit,
      activeFilter
    ]
  );

  /**
   * Reset search when query changes
   */
  const onSearchChange = (value) => {
    setSearchQuery(value);
    setUsers([]);
    setNextCursor(null);
    setHasNext(false);
  };

  return {
    users,
    loading,
    hasNext,
    fetchMore: () => fetchSearchUsers(false),
    search: () => fetchSearchUsers(true),
    onSearchChange,
    setActiveFilter
  };
}
