/**
 * adminAuth.ts
 * Centralized Administrative Service for Indzs Admin Panel
 */

const BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL || process.env.NEXT_PUBLIC_ADMIN_API_BASE_URL;

// --- Admin Specific Interfaces ---

export interface AdminUser {
    id: string;
    username: string;
    email: string;
    role: 'superadmin' | 'editor' | 'viewer';
    permissions: string[];
    last_login?: string;
}

export interface AdminAuthResponse<T = any> {
    success: boolean;
    status?: number;
    message?: string;
    data?: T;
}

// --- Internal State for Refresh Queueing ---
let isRefreshing = false;
let refreshSubscribers: ((token: string) => void)[] = [];

const subscribeTokenRefresh = (cb: (token: string) => void) => {
    refreshSubscribers.push(cb);
};

const onRefreshSuccess = (token: string) => {
    refreshSubscribers.forEach((cb) => cb(token));
    refreshSubscribers = [];
};


// --- Device Helpers ---

const getDeviceInfo = (): string => {
    if (typeof window === "undefined") return "Server Side";
    return `Platform: ${navigator.platform}, UA: ${navigator.userAgent}`;
};

const getOrCreateDeviceId = (): string => {
    if (typeof window === "undefined") return "";
    let id = localStorage.getItem("device_id");
    if (!id) {
        id = crypto.randomUUID?.() || Math.random().toString(36).substring(2);
        localStorage.setItem("device_id", id);
    }
    return id;
};

// --- Admin API Service ---

export const adminAuthService = {

    /**
     * Standardizes responses for the Admin UI
     */
    async handleResponse<T>(res: Response): Promise<AdminAuthResponse<T>> {
        const isJson = res.headers.get("content-type")?.includes("application/json");
        const body = isJson ? await res.json() : null;

        return {
            success: res.ok,
            status: res.status,
            message: body?.message || body?.error || (res.ok ? "Success" : "Admin request failed"),
            data: body,
        };
    },

    async authenticatedFetch(endpoint: string, options: RequestInit = {}): Promise<Response> {
        const url = `${BASE_URL}${endpoint}`;
        const headers = new Headers(options.headers);

        // Only set JSON content type if body is not FormData
        if (!(options.body instanceof FormData) && !headers.has("Content-Type")) {
            headers.set("Content-Type", "application/json");
        }

        if (typeof window !== "undefined") {
            headers.set("X-Device-Id", getOrCreateDeviceId());
            headers.set("X-Device-Info", getDeviceInfo());
        }

        const config = {
            ...options,
            headers,
            credentials: "include" as RequestCredentials
        };

        let response = await fetch(url, config);

        // If Access Token is expired
        if (response.status === 401) {
            if (!isRefreshing) {
                isRefreshing = true;

                const refreshResult = await this.refreshToken();

                if (refreshResult.success) {
                    isRefreshing = false;
                    onRefreshSuccess(refreshResult.data?.accessToken || "");
                    return fetch(url, config); // Retry original request
                } else {
                    isRefreshing = false;
                    return response;
                }
            }

            // Wait for existing refresh process
            return new Promise((resolve) => {
                subscribeTokenRefresh(() => {
                    resolve(fetch(url, config));
                });
            });
        }

        return response;
    },


    async refreshToken(): Promise<AdminAuthResponse<{ accessToken: string }>> {
        try {
            const res = await fetch(`${BASE_URL}/api/auth/refresh/`, {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-Device-Id": getOrCreateDeviceId(),
                },
                credentials: "include",
            });

            const data = await res.json();
            if (!res.ok) throw new Error("Session expired");
            return { success: true, data };
        } catch (error) {
            return { success: false, message: "Session expired" };
        }
    },


    async getDashboardOverview(): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch("/api/admin/dashboard/overview/", {
            method: "GET",
        });
        return this.handleResponse(res);
    },

    async getDashboardTrends(view: string, range?: { from?: string; to?: string }): Promise<AdminAuthResponse<any>> {
        const params = new URLSearchParams();

        if (view === "custom") {
            if (range?.from) params.append("from", range.from);
            if (range?.to) params.append("to", range.to);
        } else {
            params.append("days", view);
        }

        const queryString = params.toString();
        const endpoint = `/api/admin/dashboard/trends/${queryString ? `?${queryString}` : ""}`;

        const res = await this.authenticatedFetch(endpoint, {
            method: "GET",
        });

        return this.handleResponse(res);
    },




    /**
     * Fetches a paginated list of users with optional filtering and cursor tracking
     */
    async getUsersList(limit: number, filter: string, cursor: string | null): Promise<AdminAuthResponse<any>> {
        const body = {
            limit,
            filter,
            ...(cursor && { cursor })
        };

        const res = await this.authenticatedFetch("/api/admin/users/latest/", {
            method: "POST",
            body: JSON.stringify(body),
        });

        return this.handleResponse(res);
    },




    async performUserAction(path: string, body: object): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch(path, {
            method: "POST",
            body: JSON.stringify(body),
        });

        return this.handleResponse(res);
    },


    async updateUserProfile(updatedData: any): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch("/api/admin/users/update/", {
            method: "POST",
            body: JSON.stringify(updatedData),
        });

        return this.handleResponse(res);
    },




    /**
     * Searches for users based on a query string with pagination and abort support
     */
    async searchUsers(
        query: string,
        limit: number,
        filter: string,
        cursor: string | null,
        signal?: AbortSignal
    ): Promise<AdminAuthResponse<any>> {
        const body = {
            query,
            limit,
            filter,
            ...(cursor && { cursor })
        };

        const res = await this.authenticatedFetch("/api/admin/users/search/", {
            method: "POST",
            body: JSON.stringify(body),
            signal, // Pass the abort signal here
        });

        return this.handleResponse(res);
    },



    /**
     * Fetches the list of all applications/services in the ecosystem
     */
    async getAppsList(): Promise<AdminAuthResponse<any[]>> {
        const res = await this.authenticatedFetch("/api/admin/apps/", {
            method: "GET",
        });

        return this.handleResponse(res);
    },

    /**
     * Fetches a list of available App IDs for form selection
     */
    async getAvailableAppIDs(): Promise<AdminAuthResponse<{ id: string }[]>> {
        const res = await this.authenticatedFetch("/api/admin/apps/available/ids", {
            method: "GET",
        });

        return this.handleResponse(res);
    },




    /**
     * Installs or registers a new application in the ecosystem
     * Handles both file uploads and metadata via FormData
     */
    async installApp(formData: FormData): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch("/api/admin/apps/install/", {
            method: "POST",
            body: formData,
            // authenticatedFetch will automatically skip setting JSON headers
            // allowing the browser to handle the multipart boundary.
        });

        return this.handleResponse(res);
    },



    /**
     * Updates an existing application's details
     * @param appId The unique system ID of the app to update
     * @param formData The Multipart data containing the updates
     */
    async updateApp(appId: string | number, formData: FormData): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch(`/api/admin/apps/update/${appId}/`, {
            method: "PATCH",
            body: formData,
            // Header logic handled by authenticatedFetch (skips JSON type for FormData)
        });

        return this.handleResponse(res);
    },


    /**
     * Uninstalls/Removes an application from the ecosystem
     * @param appId The unique system ID of the app to delete
     */
    async uninstallApp(appId: string | number): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch(`/api/admin/apps/${appId}/uninstall/`, {
            method: "DELETE",
        });

        return this.handleResponse(res);
    },


    /**
     * Rotates the client secret for a specific application
     * @param appId The unique system ID of the app
     */
    async rotateAppSecret(appId: string | number): Promise<AdminAuthResponse<{ client_secret: string }>> {
        const res = await this.authenticatedFetch(`/api/admin/apps/${appId}/rotate-secret/`, {
            method: "POST",
        });

        return this.handleResponse(res);
    },


    /**
     * Creates a new user account from the admin panel
     * @param userData The new user details (e.g., { email, password, first_name, role })
     */
    async createUser(userData: any): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch("/api/admin/users/create/", {
            method: "POST",
            body: JSON.stringify(userData),
        });

        return this.handleResponse(res);
    },





    async getAuditActions(): Promise<AdminAuthResponse<{ total: number, actions: { key: string, value: string }[] }>> {
        const res = await this.authenticatedFetch("/api/admin/audit-logs/available/actions/", {
            method: "GET",
        });
        return this.handleResponse(res);
    },


    async getAuditLogs(params: {
        limit: number;
        cursor?: string | null;
        duration?: "7d" | "15d" | "30d" | "90d" | null;
        from_date?: string;
        to_date?: string;
        filter?: string;
    }): Promise<AdminAuthResponse<any>> {
        const body: any = {
            limit: params.limit,
        };

        // Add pagination cursor if it exists
        if (params.cursor) body.cursor = params.cursor;
        // Add Filter
        if (params.filter) body.filter = params.filter;
        // Use duration preset if provided, otherwise check for custom dates
        if (params.duration) {
            body.duration = params.duration;
        } else {
            if (params.from_date) body.from_date = params.from_date;
            if (params.to_date) body.to_date = params.to_date;

        }

        console.log(body)

        const res = await this.authenticatedFetch("/api/admin/audit-logs/latest/", {
            method: "POST",
            body: JSON.stringify(body),
        });

        return this.handleResponse(res);
    },



    async searchAuditLogs(params: {
        query: string;
        limit: number;
        cursor?: string | null;
        duration?: string | null;
        from_date?: string;
        to_date?: string;
        filter?: string;
        signal?: AbortSignal; // For debouncing/cancelling previous requests
    }): Promise<AdminAuthResponse<any>> {
        const body: any = {
            query: params.query,
            limit: params.limit,
        };

        if (params.cursor) body.cursor = params.cursor;

        if (params.filter) body.filter = params.filter;

        if (params.duration) {
            body.duration = params.duration;
        } else {
            if (params.from_date) body.from_date = params.from_date;
            if (params.to_date) body.to_date = params.to_date;
        }

        const res = await this.authenticatedFetch("/api/admin/audit-logs/search/", {
            method: "POST",
            body: JSON.stringify(body),
            signal: params.signal
        });

        return this.handleResponse(res);
    },



    async exportAuditLogsCSV(params: {
        duration?: string | null;
        from_date?: string;
        to_date?: string;
        filter?: string;
    }): Promise<void> {
        const body: any = {};

        if (params.filter) body.filter = params.filter;

        if (params.duration) {
            body.duration = params.duration;
        } else if (params.from_date && params.to_date) {
            body.from_date = params.from_date;
            body.to_date = params.to_date;
        }

        const res = await this.authenticatedFetch("/api/admin/audit-logs/export-csv/", {
            method: "POST",
            body: JSON.stringify(body),
        });

        if (!res.ok) {
            const errorData = await res.json().catch(() => ({}));
            throw new Error(errorData.message || "Export failed");
        }

        // Process the stream as a Blob
        const blob = await res.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement("a");

        // Create a filename with current date
        const dateStr = new Date().toISOString().split('T')[0];
        a.href = url;
        a.download = `audit-logs-${dateStr}.csv`;

        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
    },



    async getAuditStats(): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch("/api/admin/audit-logs/dashboard-stats/", {
            method: "GET",
        });
        return this.handleResponse(res);
    },


    /**
     * Fetch a single log entry by its unique ID
     */
    async getLogDetail(logId: string): Promise<AdminAuthResponse<any>> {
        const res = await this.authenticatedFetch(`/api/admin/audit-logs/log/${logId}/`, {
            method: "GET",
        });
        return this.handleResponse(res);
    }


};