/**
 * auth.ts
 * Centralized Authentication Service for Indzs Account
 */

const BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL;

// --- Interfaces ---

export interface User {
    id: string;
    first_name: string;
    last_name?: string;
    email: string;
    phone_number: string;
    isVerified: boolean;
    primaryPhoneVerified?: boolean;
    recovery_phone?: string;
    recovery_phone_verified?: boolean;
    primary_email?: string;
    primary_email_verified?: boolean;
    recovery_email?: string;
    recovery_email_verified?: boolean;
    dob?: string;
    gender?: string;
    profile_picture?: string;
    address?: string;
    connected_apps?: [];
    is_admin?:boolean;
    country_code: string,
    recovery_phone_country_code? : string
}

export interface Session {
    deviceId: string;
    deviceInfo: string;
    lastActive: string;
    isCurrentDevice: boolean;
}

export interface AuthResponse<T = any> {
    success: boolean;
    message?: string;
    data?: T;
    unverified?: boolean;
}

// --- Device Helpers ---

const getDeviceInfo = (): string => {
    if (typeof window === "undefined") return "Server Side";
    return `Platform: ${navigator.platform}, UA: ${navigator.userAgent}`;
};

const getOrCreateDeviceId = (): string => {
    if (typeof window === "undefined") return "";
    let id = localStorage.getItem("device_id");
    if (!id) {
        id = crypto.randomUUID?.() || Math.random().toString(36).substring(2);
        localStorage.setItem("device_id", id);
    }
    return id;
};

// --- State for Refresh Queueing ---

let isRefreshing = false;
let refreshSubscribers: ((token: string) => void)[] = [];

const subscribeTokenRefresh = (cb: (token: string) => void) => {
    refreshSubscribers.push(cb);
};

const onRefreshSuccess = (token: string) => {
    refreshSubscribers.forEach((cb) => cb(token));
    refreshSubscribers = [];
};

// --- API Service ---

export const authService = {

    /**
     * WRAPPER TO STANDARDIZE RESPONSES
     */
    async handleResponse<T>(res: Response): Promise<AuthResponse<T>> {
        const isJson = res.headers.get("content-type")?.includes("application/json");
        const body = isJson ? await res.json() : null;
        console.log(body[0])
        return {
            success: res.ok,
            status: res.status,
            message: body?.message || body?.error || body[0]|| (res.ok ? "Success" : "Request failed"),
            data: body,
            unverified: res.status === 403 || body?.unverified === true,
        };
    },


    /**
     * CORE AUTHENTICATED FETCH
     * Handles 401 Unauthorized by attempting a token refresh and retrying the request.
     */
    async authenticatedFetch(endpoint: string, options: RequestInit = {}): Promise<Response> {
        const url = `${BASE_URL}${endpoint}`;
        const headers = new Headers(options.headers);

        // Only set JSON content type if body is not FormData
        if (!(options.body instanceof FormData) && !headers.has("Content-Type")) {
            headers.set("Content-Type", "application/json");
        }

        if (typeof window !== "undefined") {
            headers.set("X-Device-Id", getOrCreateDeviceId());
            headers.set("X-Device-Info", getDeviceInfo());
        }

        const config = {
            ...options,
            headers,
            credentials: "include" as RequestCredentials
        };

        let response = await fetch(url, config);

        // If Access Token is expired
        if (response.status === 401) {
            if (!isRefreshing) {
                isRefreshing = true;

                const refreshResult = await this.refreshToken();

                if (refreshResult.success) {
                    isRefreshing = false;
                    onRefreshSuccess(refreshResult.data?.accessToken || "");
                    return fetch(url, config); // Retry original request
                } else {
                    isRefreshing = false;
                    return response;
                }
            }

            // Wait for existing refresh process
            return new Promise((resolve) => {
                subscribeTokenRefresh(() => {
                    resolve(fetch(url, config));
                });
            });
        }

        return response;
    },

    // --- 1. AUTHENTICATION & SESSION ---

    async refreshToken(): Promise<AuthResponse<{ accessToken: string }>> {
        try {
            const res = await fetch(`${BASE_URL}/api/auth/refresh/`, {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-Device-Id": getOrCreateDeviceId(),
                },
                credentials: "include",
            });

            const data = await res.json();
            if (!res.ok) throw new Error("Session expired");
            return { success: true, data };
        } catch (error) {
            return { success: false, message: "Session expired" };
        }
    },

    async login(identifier: string, password: string): Promise<Response> {
        return this.authenticatedFetch("/api/auth/login/", {
            method: "POST",
            body: JSON.stringify({ identifier, password }),
        });
    },

    async logout(): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/logout/", { method: "POST" });
        return res.json();
    },

    // async logoutAllDevices(): Promise<AuthResponse> {
    //     const res = await this.authenticatedFetch("/api/auth/logout-all/", { method: "POST" });
    //     return res.json();
    // },

    async logoutSpecificDevice(deviceId: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/logout-device/", {
            method: "POST",
            body: JSON.stringify({ deviceId }),
        });
        return res.json();
    },


    async logoutAccount(): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/logout-account/", {
            method: "POST",
            // The X-Device-Id is automatically added by authenticatedFetch
            // The refresh_token cookie is automatically sent via credentials: "include"
        });
        return this.handleResponse(res);
    },

    // --- 2. REGISTRATION, FIND ACCOUNT & RECOVERY ---

    async initiateRegister(data: Partial<User> & { password?: string }): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/register/", {
            method: "POST",
            body: JSON.stringify(data),
        });
        return res.json();
    },

    async verifyRegisterOtp(phone: string, otp: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/register/verify/", {
            method: "POST",
            body: JSON.stringify({ phone, otp }),
        });
        return res.json();
    },

    async findAccount(identifier: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/find-account/", {
            method: "POST",
            body: JSON.stringify({ identifier }),
        });
        return res.json();
    },

    async verifyFindAccountOtp(phone: string, otp: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/find-account/verify-otp/", {
            method: "POST",
            body: JSON.stringify({ phone, otp }),
        });
        return res.json();
    },




    // --- 3. PROFILE MANAGEMENT ---

    async getUserInfo(): Promise<AuthResponse<User>> {
        const res = await this.authenticatedFetch("/api/auth/me/");
        return res.json();
    },

    async updateProfile(profileData: Partial<User>): Promise<AuthResponse<User>> {
        const res = await this.authenticatedFetch("/api/accounts/update-profile/", {
            method: "PATCH",
            body: JSON.stringify(profileData),
        });
        return this.handleResponse<User>(res);;
    },


    async uploadProfilePicture(formData: FormData): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/profile/upload-picture/", {
            method: "POST",
            body: formData,
            // IMPORTANT: We leave headers empty here. 
            // authenticatedFetch is programmed to NOT set Content-Type 
            // when body is FormData, allowing the browser to set the boundary.
            headers: {},
        });
        return this.handleResponse(res);
    },

    async removeProfilePicture(): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/profile/remove-picture/", {
            method: "DELETE", // Change to "DELETE" if your Django path requires it
        });
        return this.handleResponse(res);
    },



    async sendRecoveryEmailOtp(identifier: string,countryCode?:string, purpose?: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/otp/recovery/identifier/send/", {
            method: "POST",
            body: JSON.stringify({ identifier,country_code:countryCode, purpose }),
        });
        return this.handleResponse(res);
    },

    async verifyRecoveryOtp(identifier: string,send_to:string,countryCode: string, otp: string, purpose?: string): Promise<AuthResponse> {
        console.log(identifier,countryCode, otp, purpose)

        const res = await this.authenticatedFetch("/api/auth/recovery/otp/verify/", {
            method: "POST",
            body: JSON.stringify({ 
                "identifier" : send_to, 
                "to_verify":identifier, 
                "country_code":countryCode,
                "otp": otp, 
                "purpose" : purpose }),
        });
        return this.handleResponse(res);
    },

    // async uploadProfilePicture(formData: FormData): Promise<AuthResponse> {
    //     const res = await this.authenticatedFetch("/api/auth/profile/upload-picture/", {
    //         method: "POST",
    //         body: formData,
    //         headers: {}, // Browser handles multipart boundary
    //     });
    //     return res.json();
    // },

    // async removeProfilePicture(): Promise<AuthResponse> {
    //     const res = await this.authenticatedFetch("/api/auth/profile/remove-picture/", {
    //         method: "POST",
    //     });
    //     return res.json();
    // },

    // --- 4. DEVICE & APP CONNECTIONS ---

    // async getLoggedInDevices(): Promise<AuthResponse<Session[]>> {
    //     const res = await this.authenticatedFetch("/api/auth/logged-in-devices/");
    //     return res.json();
    // },

    // async connectApp(appId: string): Promise<AuthResponse> {
    //     const res = await this.authenticatedFetch(`/api/auth/connect/${appId}/`, {
    //         method: "POST",
    //     });
    //     return res.json();
    // },

    // async disconnectApp(appId: string): Promise<AuthResponse> {
    //     const res = await this.authenticatedFetch(`/api/auth/disconnect/${appId}/`, {
    //         method: "POST",
    //     });
    //     return res.json();
    // },



    async connectApp(appId: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch(`/api/accounts/connect/${appId}/`, {
            method: "POST",
        });
        return this.handleResponse(res);
    },

    async disconnectApp(appId: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch(`/api/accounts/disconnect/${appId}/`, {
            method: "POST",
        });
        return this.handleResponse(res);
    },
    // Inside authService object in auth.ts

    async getLoggedInDevices(): Promise<AuthResponse<Session[]>> {
        const res = await this.authenticatedFetch("/api/auth/logged-in-devices/", {
            method: "POST", // Your code used POST for fetching devices
        });
        return this.handleResponse(res);
    },

    async logoutDevice(deviceId: string, sessionId: string): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/logout-device/", {
            method: "POST",
            body: JSON.stringify({ device_id: deviceId, session_id: sessionId }),
        });
        return this.handleResponse(res);
    },

    async logoutAllDevices(): Promise<AuthResponse> {
        const res = await this.authenticatedFetch("/api/auth/logout-all/", {
            method: "POST",
        });
        return this.handleResponse(res);
    },

    
    async getAvailableApps(): Promise<AuthResponse<any[]>> {
        const res = await this.authenticatedFetch("/api/accounts/apps/available/", {
            method: "GET",
        });
        return this.handleResponse(res);
    },

};